#pragma once

#include <iostream>
#include <utility>
#include <memory>
#include <boost/fusion/algorithm.hpp>
#include <boost/fusion/adapted.hpp>
#include <yplatform/tskv/detail/attribute.h>

namespace yplatform {
namespace tskv {
namespace detail {

struct endl_t {};

template <typename Stream>
struct out_attr
{
    Stream& s;
    template <typename K, typename V>
    void operator() (const attribute<K, V>& v) const
    { s << v;}
};

struct finalizator {};

template <typename T>
using is_finalizator = std::is_base_of<finalizator, T>;

struct finalize_with_endl : finalizator
{
    template <typename Ostream>
    void operator()(Ostream* s) const
    {
        if (s) {
            *s << std::endl;
        }
    }
};

struct finalize_without_endl : finalizator
{
    template <typename Ostream>
    void operator()(Ostream*) const {}
};

/**
 * Stream wrapper - represents the TSKV log line. It closes line on destruction
 * or mandatory with tskv::endl manipulator. Please note what after tskv::endl
 * manipulator is applied steam_wrapper can not be used, because of undefined
 * behavior.
 */
template <typename Ostream, typename Converters, typename Finalization>
class stream_wrapper
{
    using ostream = Ostream;
    using char_type = typename ostream::char_type;
    using traits_type = typename ostream::traits_type;

    constexpr static const char_type* tskv_head = "tskv";
    constexpr static const char_type* attr_sep = "\t";
    constexpr static const char_type* kv_sep = "=";

    std::unique_ptr<ostream, Finalization> s;
    Converters converters;

public:
    stream_wrapper(ostream& s, Converters conv, Finalization fin)
    : s(&s, std::move(fin)), converters(std::move(conv))
    { s << tskv_head; }

    template <typename Key, typename Value>
    typename std::enable_if<!is_attributes_sequence<attribute<Key, Value>>::value,
                stream_wrapper>::type& operator << (const attribute<Key, Value>& v)
    {
        *s << attr_sep;
        convert_key(*s, converters, key(v));
        *s << kv_sep;
        convert_value(*s, converters, value(v));
        return *this;
    }

    template <typename Sequence>
    friend typename std::enable_if<is_attributes_sequence<Sequence>::value,
            stream_wrapper>::type& operator << (stream_wrapper& w, const Sequence& v)
    {
        boost::fusion::for_each(v, out_attr<stream_wrapper>{w});
        return w;
    }

    void operator << (const endl_t&)
    { s.reset(); }

    template <typename Key, typename Value>
    stream_wrapper& operator << (const std::pair<Key, Value>& v)
    { return *this << std::tie(v.first, v.second); }

    template <typename Container>
    friend typename std::enable_if<is_attributes_container<Container>::value,
                stream_wrapper>::type& operator << (stream_wrapper& w, const Container& c)
    {
        for (const auto& attr : c) {
            w << attr;
        }
        return w;
    }
};

template <typename V, typename ...Ts>
inline stream_wrapper<Ts...> operator << (stream_wrapper<Ts...>&& w, const V& v)
{
    w << v;
    return std::move(w);
}

} // namespace detail
} // namespace tskv
} // namespace yplatform
