#pragma once

#include <iostream>
#include <limits>
#include <boost/format.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/locale/encoding_utf.hpp>
#include <yplatform/tskv/detail/tweak.h>
#include <yplatform/tskv/detail/basic_converter.h>

namespace yplatform {
namespace tskv {
namespace detail {

/**
 * Utf converter. Assumes that your input string is utf encoded. All illegal
 * characters will be omitted. All non-printable characters according to the
 * output stream's locale will be escaped. So pay attention on output stream's
 * locale when use this converter.
 * Please note what filter is not applicable to the numerics, boolean and void
 * pointer.
 */
struct utf_converter : public basic_converter
{
    struct escaper
    {
        template <class T>
        std::string operator()(T&& match) const
        {
            std::string s;
            for (const auto chr : match) {
                s += escape(chr);
            }
            return s;
        }

        static std::string escape(const wchar_t chr)
        {
            switch (chr) {
                case '\n':
                    return "\\n";
                case '\t':
                    return "\\t";
                case '\r':
                    return "\\r";
                case '\\':
                    return "\\\\";
                default:
                    return str(format(chr) % static_cast<int>(chr));
            }
        }

        static boost::format format(const wchar_t chr)
        {
            if (fits<char>(chr)) {
                return boost::format("\\x%02x");
            } else if (fits<char16_t>(chr)) {
                return boost::format("\\u%04x");
            } else {
                return boost::format("\\U%08x");
            }
        }

        template <class T>
        static bool fits(const wchar_t chr)
        {
            return static_cast<wchar_t>(std::numeric_limits<T>::min()) <= chr
                && chr <= static_cast<wchar_t>(std::numeric_limits<T>::max());
        }
    };

    escaper escape;

    template <typename Ostream, typename T>
    Ostream& operator()(Ostream& s, T& v) const
    {
        return (*this)(s, boost::lexical_cast<string_type<Ostream>>(v));
    }

    __YPLATFORM_TSKV_TWEAK_BYPASS_POD_NUMERIC_TYPES

    template <typename Ostream>
    Ostream& operator()(Ostream& s, const string_type<Ostream>& str) const
    {
        using namespace boost;
        using namespace boost::locale::conv;

        auto unicode_str = utf_to_utf<wchar_t>(str);

        const auto printable = is_print(s.getloc());
        const auto must_be_escaped = [&printable](wchar_t ch) { return !printable(ch) || ch == '\\'; };

        find_format_all(unicode_str, token_finder(must_be_escaped), escape);
        s << utf_to_utf<typename Ostream::char_type>(unicode_str);
        return s;
    }
};

} // namespace detail
} // namespace tskv
} // namespace yplatform
