/*
 * ArjReader.cpp
 *
 *  Created on: 17 июл. 2017 г.
 *      Author: luckybug
 */

#include <util/stream/input.h>
#include <util/generic/buffer.h>
#include <util/stream/mem.h>

#include "ArjReader.h"

namespace NArj {
    /*
     * breath
     * Actually this class only reads file info in the archive, not de/compress it
     * http://www.opennet.ru/docs/formats/arj.txt
     */
    
    bool ReadMainHeaderAndData(IInputStream & stream);
    bool ReadLocalHeaderAndData(IInputStream & stream, TFileTraits& traits);
    bool SearchHeaderId(IInputStream & stream);
    bool ReadFirstHeader(IInputStream & stream, TFileTraits& traits, ui32 & compressedSize);
    bool SkipFirstHeader(IInputStream & stream);
    bool SkipString(IInputStream & stream);

    bool ReadTraits(IInputStream & stream, TTraitsList& items) {
        if (!ReadMainHeaderAndData(stream))
            return false;

        TFileTraits traits;
        while (ReadLocalHeaderAndData(stream, traits)) {
            items.push_back(traits);
        }

        return true;
    }

    bool ReadTraits(const char* data, size_t size, TTraitsList& items) {
        TMemoryInput stream(data, size);

        return ReadTraits(stream, items);
    }

    bool ReadMainHeaderAndData(IInputStream & stream) {
        if (!SearchHeaderId(stream))
            return false;

        if(stream.Skip(2) != 2) // basic header size
            return false;

        if (!SkipFirstHeader(stream))
            return false;

        if (!SkipString(stream)) //    filename
            return false;

        if (!SkipString(stream)) //    archive comment
            return false;

        if(stream.Skip(4) != 4) // basic header CRC
            return false;

        ui16 extendedHeaderSize = {};
        if(!stream.Read(&extendedHeaderSize, 2))
            return false;

        if (extendedHeaderSize) {
            stream.Skip(extendedHeaderSize); // 1st extended header (currently not used)
            stream.Skip(4); //1st extended header's CRC (not present when 0 extended header size)
        }

        return true;
    }

    bool ReadLocalHeaderAndData(IInputStream & stream, TFileTraits& traits) {
        if (!SearchHeaderId(stream))
            return false;

        if(stream.Skip(2) != 2) // basic header size
            return false;

        ui32 compressedSize = {};
        if (!ReadFirstHeader(stream, traits, compressedSize))
            return false;

        if(!stream.ReadTo(traits.filename, '\0'))
            return false;

        if (!SkipString(stream)) //    comment
            return false;

        if(stream.Skip(4) != 4) // basic header CRC
            return false;

        ui16 extendedHeaderSize = {};
        if(!stream.Read(&extendedHeaderSize, 2))
            return false;

        if (extendedHeaderSize) {
            if(stream.Skip(extendedHeaderSize) != extendedHeaderSize || stream.Skip(4) != 4)
                return false;
        }

        return stream.Skip(compressedSize) == compressedSize;
    }


    bool SearchHeaderId(IInputStream & stream) {
        char c1, c2;

        if(!stream.ReadChar(c1))
            return false;

        while (stream.ReadChar(c2)) {
            if (static_cast<ui8>(c1) == 0x60 && static_cast<ui8>(c2) == 0xea)
                return true;
            c1 = c2;
        }

        return false;
    }

    bool ReadFirstHeader(IInputStream & stream, TFileTraits& traits, ui32 & compressedSize) {
        ui8 firstHeaderSize = {};
        if (!stream.ReadChar((char&)firstHeaderSize))
            return false;

        Cerr << "header size " << (int)firstHeaderSize << Endl;

        static const size_t compressedSizeOffset = 11;

        if(stream.Skip(compressedSizeOffset) != compressedSizeOffset || !stream.Read(&compressedSize, 4))
            return false;

        if(!stream.Read(&traits.size, 4))
            return false;

        if(!stream.Read(&traits.crc, 4))
            return false;

        const size_t toSkip = firstHeaderSize - compressedSizeOffset - 12;

        return stream.Skip(toSkip) == toSkip;
    }

    bool SkipFirstHeader(IInputStream & stream) {
        ui8 firstHeaderSize = {};
        return !(!stream.ReadChar((char&)firstHeaderSize) && !stream.Skip(static_cast<size_t>(firstHeaderSize)));

    }

    bool SkipString(IInputStream & stream) {
        char c;
        do {
            if(!stream.ReadChar(c))
                return false;
        } while(c != '\0');
        return true;
    }

} //    namespace NArj
