#include <maps/wikimap/mapspro/libs/acl/include/subject_path.h>
#include <maps/wikimap/mapspro/libs/acl/include/exception.h>

#include <yandex/maps/wiki/common/string_utils.h>
#include <boost/algorithm/string/split.hpp>

namespace maps::wiki::acl {

namespace {

const char PATH_DELIMITER = '/';

std::list<std::string>
splitPath(const std::string& path)
{
    std::list<std::string> parts;
    boost::split(parts, path, [](char a){ return a == PATH_DELIMITER; });
    return parts;
}

bool
startsWith(const std::string& text, const std::string& subText)
{
    return std::equal(subText.begin(), subText.end(), text.begin());
}

bool
equalUpToDelimiter(const std::string& text, const std::string& subText)
{
    if (subText.size() > text.size()) {
        return false;
    }
    if (!startsWith(text, subText)) {
        return false;
    }
    if (subText.size() == text.size()) {
        return true;
    }
    return text[subText.size()] == PATH_DELIMITER;
}

bool
pathAllowsResource(const std::string& path, const std::string& resource)
{
    return equalUpToDelimiter(resource, path);
}

bool
pathAllowsResourcePartially(const std::string& path, const std::string& resource)
{
    return equalUpToDelimiter(resource, path) || equalUpToDelimiter(path, resource);
}

} // namespace

SubjectPath::SubjectPath()
{
}

SubjectPath::SubjectPath(const std::string& root)
{
    if (root.empty()) {
        throw BadParam() << "Empty permission path root";
    }
    std::istringstream is(root);
    is >> *this;
}

SubjectPath
SubjectPath::operator()(const std::string& level) const
{
    if (level.empty()) {
        throw BadParam() << "Empty permission path level";
    }
    SubjectPath step;
    step.pathParts_ = pathParts_;
    auto parts = splitPath(level);
    step.pathParts_.insert(step.pathParts_.end(), parts.begin(), parts.end());
    return step;
}

bool
SubjectPath::isAllowed(const CheckContext& context) const
{
    auto resource = str();
    for (const auto& path : context.allowedPaths_) {
        if (pathAllowsResource(path, resource)) {
            return true;
        }
    }
    return false;
}

bool
SubjectPath::isAllowedPartially(const CheckContext& context) const
{
    auto resource = str();
    for (const auto& path : context.allowedPaths_) {
        if (pathAllowsResourcePartially(path, resource)) {
            return true;
        }
    }
    return false;
}

void
SubjectPath::check(const CheckContext& context) const
{
    if (!isAllowed(context)) {
        throw AccessDenied(context.uid_) << " partial access denied to " << str();
    }
}

void
SubjectPath::checkPartialAccess(const CheckContext& context) const
{
    if (!isAllowedPartially(context)) {
        throw AccessDenied(context.uid_) << " access denied to " << str();
    }
}

std::string
SubjectPath::str() const
{
    return common::join(pathParts_, PATH_DELIMITER);
}

std::istream&
operator>>(std::istream& is, SubjectPath& path)
{
    std::string readString;
    is >> readString;
    std::list<std::string> parts = splitPath(readString);
    for (const auto& part : parts) {
        path = path(part);
    }
    return is;
}

} // namespace maps::wiki::acl
