#include <maps/wikimap/mapspro/libs/acl_utils/include/feedback_preset_checker.h>

#include <maps/wikimap/mapspro/libs/acl/include/aclgateway.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/preset.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/task.h>
#include <maps/wikimap/mapspro/libs/social/include/yandex/maps/wiki/social/feedback/task_aoi.h>

#include <algorithm>

namespace maps::wiki::acl_utils {

namespace {

struct PresetAndRole
{
    sf::Preset preset;
    sf::RoleKind roleKind;
};

AssignedPresetsExt getAssignedPresetsExt(
    pgpool3::Pool& corePool,
    pgpool3::Pool& socialPool,
    social::TUid uid,
    std::chrono::steady_clock::time_point timePoint)
{
    const auto presets = [&] {
        auto socialTxn = socialPool.slaveTransaction();
        return sf::getPresets(*socialTxn);
    }();

    std::map<social::TId, PresetAndRole> roleIdToPresetAndRole;
    for (const auto& preset : presets) {
        for (auto roleKind : enum_io::enumerateValues<sf::RoleKind>()) {
            roleIdToPresetAndRole[preset.roles.getId(roleKind)] = PresetAndRole{
                .preset = preset,
                .roleKind = roleKind,
            };
        }
    }

    auto coreTxn = corePool.slaveTransaction();
    const auto policies = acl::ACLGateway(*coreTxn).user(uid).allPolicies();

    AssignedPresetsExt retVal(timePoint);
    for (const auto& policy: policies) {
        const auto indexIt = roleIdToPresetAndRole.find(policy.roleId());
        if (indexIt == roleIdToPresetAndRole.end()) {
            continue;
        }
        retVal.add(
            indexIt->second.preset,
            indexIt->second.roleKind,
            policy.aoiId() ? std::make_optional(policy.aoiId()) : std::nullopt);
    }
    return retVal;
};

template <typename T>
bool contains(const std::vector<T>& arr, const T& value) {
    return std::find(arr.begin(), arr.end(), value) != arr.end();
}

bool matched(PresetRightKind rightKind, sf::RoleKind roleKind)
{
    switch (rightKind) {
        case PresetRightKind::Modify:
            if (roleKind == sf::RoleKind::AllRights) {
                return true;
            }
            break;
        case PresetRightKind::View:
            return true;
    }
    return false;
}

} // unnamed namespace

namespace internal {

bool match(const sf::TaskBrief& task, const sf::PresetEntries& entries)
{
    if (entries.hidden && entries.hidden.value() != task.hidden()) {
        return false;
    }
    if (entries.types && !contains(*entries.types, task.type())) {
        return false;
    }
    if (entries.sources && !contains(*entries.sources, task.source())) {
        return false;
    }
    if (entries.workflows && !contains(*entries.workflows, task.workflow())) {
        return false;
    }
    return true;
}

}  // namespace internal

AssignedPreset::AssignedPreset(
    social::TId presetId,
    sf::RoleKind roleKind,
    std::optional<social::TId> aoiId)
    : presetId(presetId)
    , roleKind(roleKind)
    , aoiId(aoiId)
{}

AssignedPresetsExt::AssignedPresetsExt(SteadyTimePoint loadedFromDB)
    : loadedFromDB_(loadedFromDB)
{}

void AssignedPresetsExt::add(sf::Preset preset, sf::RoleKind roleKind, std::optional<social::TId> aoiId)
{
    assignedPresets_.emplace_back(preset.id, roleKind, aoiId);
    presetsById_.emplace(preset.id, std::move(preset));
}

std::vector<AssignedPreset> AssignedPresetsExt::getAssignedPresets(PresetRightKind rightKind) const
{
    std::vector<AssignedPreset> retVal;
    for (const auto& assignedPreset : assignedPresets_) {
        if (matched(rightKind, assignedPreset.roleKind)) {
            retVal.push_back(assignedPreset);
        }
    }
    return retVal;
}

const sf::Preset& AssignedPresetsExt::requirePreset(social::TId presetId) const
{
    auto it = presetsById_.find(presetId);
    REQUIRE(it != presetsById_.end(), "preset Id not found" << presetId);
    return it->second;
}

SteadyTimePoint AssignedPresetsExt::loadedFromDB() const
{
    return loadedFromDB_;
}

bool FeedbackPresetCheckerImpl::isInAssignedPreset(
    acl::UID uid, const sf::Task& task,
    PresetRightKind rightKind) const
{
    auto assignedExt = getAssignedPresets(uid);

    auto socialTxn = socialPool_.slaveTransaction();
    social::TIds aois = sf::getTasksAoiIds(*socialTxn, {task.id()})[task.id()];

    // check task against every AssignedPreset in collection
    for (const auto& assigned : assignedExt.getAssignedPresets(rightKind)) {
        if (assigned.aoiId && !aois.contains(*assigned.aoiId)) {
            continue;
        }
        const sf::Preset& preset = assignedExt.requirePreset(assigned.presetId);
        if (internal::match(task, preset.entries)) {
            return true;
        }
    }
    return false;
}

sf::Presets FeedbackPresetCheckerImpl::getGlobalPresets(acl::UID uid, PresetRightKind rightKind) const
{
    sf::Presets globalPresets;
    auto assignedExt = getAssignedPresets(uid);
    for (const auto& assigned : assignedExt.getAssignedPresets(rightKind)) {
        if (!assigned.aoiId.has_value()) {
            globalPresets.emplace_back(assignedExt.requirePreset(assigned.presetId));
        }
    }
    return globalPresets;
}

void FeedbackPresetCheckerImpl::restrictFilterByGlobalPresets(sf::MvSourceTypeFilter& filter, social::TUid uid) const
{
    auto globalPresets = getGlobalPresets(uid, PresetRightKind::View);
    if (globalPresets.empty()) {
        filter.types(sf::Types{});  // restrict ALL
        return;
    }
    for (const auto& preset : globalPresets) {
        filter.addBaseDimensions(preset.getBaseDimensions());
    }
}

void FeedbackPresetCheckerImpl::restrictFilterByGlobalPresets(sf::TaskFilter& filter, social::TUid uid) const
{
    auto globalPresets = getGlobalPresets(uid, PresetRightKind::View);
    if (globalPresets.empty()) {
        filter.types(sf::Types{});  // restrict ALL
        return;
    }
    for (const auto& preset : globalPresets) {
        filter.addBaseDimensions(preset.getBaseDimensions());
    }
}

bool FeedbackPresetCheckerImpl::hasModifyPermission(
    acl::UID uid,
    const sf::Preset& preset,
    social::TId aoiId) const
{
    auto coreTxn = corePool_.slaveTransaction();
    acl::ACLGateway aclGateway(*coreTxn);

    const auto policies = aclGateway.user(uid).allPolicies();

    for (const auto& policy: policies) {
        if (policy.roleId() != preset.roles.getId(sf::RoleKind::AllRights)) {
            continue;
        }
        if (aoiId == policy.aoiId()) {
            return true;
        }
    }
    return false;
}

AssignedPresetsExt
FeedbackPresetCheckerImpl::getAssignedPresets(social::TUid uid) const
{
    const auto now = std::chrono::steady_clock::now();

    auto evalAssignedPresetsExt = [&](social::TUid uid) {
        return getAssignedPresetsExt(corePool_, socialPool_, uid, now);
    };

    const auto& assignedPresetsExt = cache_.getOrEmplace(uid, evalAssignedPresetsExt);

    REQUIRE(assignedPresetsExt, "null value from lru-cache");
    if (now - assignedPresetsExt->loadedFromDB() < expirationPeriod_) {
        return *assignedPresetsExt;
    }

    return *cache_.put(uid, evalAssignedPresetsExt(uid));
}

} // namespace maps::wiki::acl_utils
