#include <maps/wikimap/mapspro/libs/assessment/include/gateway.h>

#include "units/load.h"
#include "units/create.h"

#include "competences/create.h"
#include "competences/load.h"

#include "samples/load.h"
#include "samples/create.h"
#include "samples/stats.h"
#include "samples/cleanup.h"

#include <yandex/maps/wiki/social/feedback/gateway_ro.h>
#include <yandex/maps/wiki/common/commit_properties.h>
#include <maps/wikimap/mapspro/libs/assessment/impl/magic_strings.h>
#include <maps/libs/enum_io/include/enum_io.h>

#include <util/string/cast.h>

#include <string_view>
#include <unordered_map>
#include <unordered_set>

namespace maps::wiki::assessment {

namespace feedback = maps::wiki::social::feedback;

namespace {

const std::unordered_map<Entity::Domain, std::unordered_set<std::string_view>> DOMAIN_TO_ACTIONS = {
    {
        Entity::Domain::Edits,
        {
            common::COMMIT_ALL_ACTIONS.cbegin(),
            common::COMMIT_ALL_ACTIONS.cend()
        }
    },
    {
        Entity::Domain::Feedback,
        {
            toString(feedback::TaskOperation::Accept),
            toString(feedback::TaskOperation::Reject),
            toString(feedback::TaskOperation::NeedInfo)
        }
    },
    {
        Entity::Domain::Moderation,
        {
            "resolve",
            "close"
        }
    },
    {
        Entity::Domain::Tracker,
        {
            TRACKER_ACTION_CLOSE,
            TRACKER_ACTION_CHANGE_STATUS_TO_ON_SUPPORT_SIDE
        }
    }
};

} // namespace

Gateway::Gateway(pqxx::transaction_base& txn)
    : txn_{txn}
{}

void Gateway::checkEntityAction(const Entity& entity, const Action& action) const
{
    REQUIRE(
        DOMAIN_TO_ACTIONS.count(entity.domain),
        "Unsupported domain '" << entity.domain << "'"
    );
    REQUIRE(
        DOMAIN_TO_ACTIONS.at(entity.domain).count(action.name),
        "Unsupported action '" << action.name << "' for domain '" << entity.domain << "'"
    );

    if (entity.domain == Entity::Domain::Feedback) {
        TId entity_id{0};
        REQUIRE(
            TryFromString(entity.id, entity_id),
            "Entity id '" << entity.id << "' cannot be converted to a number."
        );
        checkFeedbackAction(entity_id, action);
    }
}

void Gateway::checkFeedbackAction(
    TId feedbackTaskId,
    const Action& action) const
{
    auto history = feedback::GatewayRO(txn_).history(feedbackTaskId);
    REQUIRE(!history.items().empty(), "No feedback task with id = " << feedbackTaskId);

    const auto operation = enum_io::fromString<feedback::TaskOperation>(action.name);
    for (const auto& item: history.items()) {
        if (item.operation() == operation &&
            item.modifiedBy() == action.by &&
            item.modifiedAt() == action.at)
        {
            return;
        }
    }
    throw RuntimeError() <<
        "No TaskOperation " << operation << " "
        "by uid = " << action.by << " "
        "at " << chrono::formatSqlDateTime(action.at) << " "
        "for feedback task with id = " << feedbackTaskId;
}

TId Gateway::getOrCreateUnit(const Entity& entity, const Action& action)
{
    auto unitId = units::get(txn_, entity, action);
    return unitId ? *unitId : units::create(txn_, entity, action);
}

TId Gateway::getOrCreateCompetence(const std::string& roleName)
{
    auto competenceId = competences::get(txn_, roleName);
    return competenceId ? *competenceId : competences::create(txn_, roleName);
}

CompetenceVec Gateway::loadCompetences(const std::vector<std::string>& roleNames) const
{
    return competences::load(txn_, roleNames);
}

TId Gateway::createSample(
    Entity::Domain domain,
    Qualification qualification,
    const std::string& name,
    TIds unitIds,
    size_t tasksPerUnit)
{
    REQUIRE(!unitIds.empty(), "Can't create an empty sample.");
    REQUIRE(tasksPerUnit > 0, "tasksPerUnit must be greater than zero.");

    const TIds NO_COMPETENCE_IDS = {};
    const auto sampleId = samples::create(txn_, domain, qualification, name);

    for (const auto unitId: unitIds) {
        for (size_t taskCounter = 0; taskCounter < tasksPerUnit; ++taskCounter) {
            samples::createTask(txn_, sampleId, unitId, NO_COMPETENCE_IDS);
        }
    }

    return sampleId;
}

SampleInfo Gateway::findSample(TId sampleId) const
{
    auto sampleInfo = samples::loadById(txn_, sampleId);
    REQUIRE(sampleInfo, "Found no sample by id = " << sampleId);

    sampleInfo->assessorsStats = samples::loadAssessorsStats(txn_, sampleId);
    return *sampleInfo;
}

UnitGradeStatsVec Gateway::sampleGradeStats(TId sampleId) const
{
    return samples::loadGradeStats(txn_, sampleId);
}

bool Gateway::sampleExistsByName(const std::string& name) const
{
    return samples::existsByName(txn_, name);
}

size_t Gateway::deleteOldUnitSkips(chrono::TimePoint olderThan)
{
    return samples::deleteOldUnitSkips(txn_, olderThan);
}

Console Gateway::console(TUid uid) const
{
    return Console(txn_, uid);
}

} // namespace maps::wiki::assessment
