#include "load.h"

#include <maps/wikimap/mapspro/libs/assessment/impl/sql_helpers.h>
#include <maps/wikimap/mapspro/libs/assessment/impl/magic_strings.h>

namespace maps::wiki::assessment::samples {

namespace {

SampleInfo makeSampleInfo(const pqxx::row& row, const std::optional<TUid> uid = {})
{
    SampleInfo sampleInfo = {
        row[sql::col::SAMPLE_ID].as<TId>(),
        enum_io::fromString<Entity::Domain>(row[sql::col::ENTITY_DOMAIN].c_str()),
        enum_io::fromString<Qualification>(row[sql::col::QUALIFICATION].c_str()),
        chrono::parseSqlDateTime(row[sql::col::CREATED_AT].c_str()),
        row[sql::col::NAME].c_str(),
        row["graded_units"].as<uint64_t>(),
        row["total_units"].as<uint64_t>(),
        {}
    };
    if (uid && !row["my_graded_units"].is_null()) {
        const auto myGraded = row["my_graded_units"].as<uint64_t>();
        if (myGraded > 0 ){
            ASSERT(!row["my_recent_graded_at"].is_null());
            sampleInfo.assessorsStats.push_back({
                *uid,
                myGraded,
                chrono::parseSqlDateTime(row["my_recent_graded_at"].c_str())});
        }
    }
    return sampleInfo;
}

namespace sqlClause {

std::string where(const SampleFeedParams& params)
{
    // newest-first order
    if (params.before()) {
        return "(" + sql::col::SAMPLE_ID + " > " + std::to_string(params.before()) + ")";
    }
    if (params.after()) {
        return "(" + sql::col::SAMPLE_ID + " < " + std::to_string(params.after()) + ")";
    }
    return "TRUE";
}

std::string where(pqxx::transaction_base& txn, const SampleFilter& filter)
{
    std::string query = "TRUE";

    if (filter.entityDomain()) {
        query += " AND " + sql::col::ENTITY_DOMAIN + " = " + sqlEnumToString(txn, *filter.entityDomain());
    }
    if (filter.qualification()) {
        query += " AND " + sql::col::QUALIFICATION + " = " + sqlEnumToString(txn, *filter.qualification());
    }
    if (filter.createdAt()) {
        query += " AND " + filter.createdAt()->sqlComparison(txn, sql::col::CREATED_AT);
    }
    for (const auto& nameLike: filter.nameLike()) {
        query += " AND " + sql::col::NAME + " LIKE " + txn.quote(nameLike);
    }
    for (const auto& nameNotLike: filter.nameNotLike()) {
        query += " AND " + sql::col::NAME + " NOT LIKE " + txn.quote(nameNotLike);
    }

    return query;
}

std::string having(const SampleFilter& filter)
{
    if (filter.graded()) {
        return
            "COUNT(" + sql::col::GRADE_ID + ") " +
            (*filter.graded() ? "=" : "<") + " "
            "COUNT(" + sql::col::UNIT_ID + ")";
    }

    return "TRUE";
}

} // namespace sqlClause


SampleInfoVec load(
    pqxx::transaction_base& txn,
    const std::optional<TUid>& uid,
    const std::string& where,
    const std::string& having,
    const std::string& orderBy,
    size_t limit)
{
    const auto uidStr = uid ? std::to_string(*uid) : "0";
    const auto result = txn.exec(
        "SELECT " +
            sql::col::SAMPLE_ID + ", " +
            sql::col::ENTITY_DOMAIN + ", " +
            sql::col::QUALIFICATION + ", " +
            sql::col::CREATED_AT + ", " +
            sql::col::NAME + ", " +
            "COUNT(" + sql::col::GRADE_ID + ") as graded_units, " +
            "COUNT(" + sql::col::UNIT_ID + ") as total_units, " +
            "COUNT("
                "CASE WHEN " +
                    sql::col::ACQUIRED_BY + " = " + uidStr + " "
                "THEN " +
                    sql::col::GRADE_ID + " END) as my_graded_units, " +
            "MAX("
                "CASE WHEN " +
                    sql::col::ACQUIRED_BY + " = " + uidStr + " AND " +
                    sql::col::GRADE_ID + " IS NOT NULL "
                "THEN " + sql::col::ACQUIRED_AT + " END) as my_recent_graded_at "
        "FROM " +
            sql::table::SAMPLE + " "
            "LEFT JOIN " + sql::table::SAMPLE_TASK + " "
            "USING (" + sql::col::SAMPLE_ID + ") "
        "WHERE " +
             where + " "
        "GROUP BY " +
            sql::col::SAMPLE_ID + " "
        "HAVING " +
            having + " "
        "ORDER BY " +
            orderBy + " "
        "LIMIT " +
            std::to_string(limit));

    SampleInfoVec samplesInfo;
    samplesInfo.reserve(result.size());
    for (const auto& row : result) {
        samplesInfo.emplace_back(makeSampleInfo(row, uid));
    }
    return samplesInfo;
}

} // namespace

std::optional<SampleInfo> loadById(
    pqxx::transaction_base& txn,
    TId sampleId)
{
    const auto where = sql::col::SAMPLE_ID + " = " + std::to_string(sampleId);
    const auto having = "TRUE";
    const auto orderBy = sql::col::SAMPLE_ID;
    const size_t limit = 1;

    const auto result = load(txn, {}, where, having, orderBy, limit);
    return result.size() ? std::make_optional(result[0]) : std::nullopt;
}

SampleFeed loadFeed(
    pqxx::transaction_base& txn,
    const std::optional<TUid>& uid,
    const SampleFeedParams& params,
    const SampleFilter& filter)
{
    const auto where = "(" + sqlClause::where(params) + ") AND (" + sqlClause::where(txn, filter) + ")";
    const auto having = sqlClause::having(filter);
    const auto orderBy = sql::col::SAMPLE_ID + (params.before() ? " ASC" : " DESC");
    const auto samplesInfo = load(txn, uid, where, having, orderBy, params.perPage() + 1);

    const auto size = std::min(samplesInfo.size(), params.perPage());
    const auto hasMore = samplesInfo.size() > params.perPage();
    if (params.before()) {
        return SampleFeed(SampleInfoVec(samplesInfo.crbegin(), samplesInfo.crbegin() + size), hasMore);
    }
    return SampleFeed(SampleInfoVec(samplesInfo.cbegin(), samplesInfo.cbegin() + size), hasMore);
}

bool existsByName(
    pqxx::transaction_base& txn,
    const std::string& name)
{
    return txn.exec(
        "SELECT EXISTS("
            "SELECT 1 "
            "FROM " + sql::table::SAMPLE + " "
            "WHERE " + sql::col::NAME + " = " + txn.quote(name) +
        ")")[0][0].as<bool>();
}

} // maps::wiki::assessment::samples
