#include "status.h"

#include <maps/wikimap/mapspro/libs/assessment/impl/magic_strings.h>
#include <maps/wikimap/mapspro/libs/assessment/impl/sql_helpers.h>

#include <fmt/format.h>

namespace maps::wiki::assessment::units {

using namespace fmt::literals;

namespace sqlClause {

namespace {

const std::string TRY_FIX_UNIT_FMT = fmt::format(
    "UPDATE {unit} "
    "SET {fixed_at} = NOW() "
    "WHERE {{fixable}} AND {unit_id} = {{unit_id}}",

    "unit"_a = sql::table::UNIT,
    "fixed_at"_a = sql::col::FIXED_AT,
    "unit_id"_a = sql::col::UNIT_ID);

const std::string OVERALL_VALUE_EQUALS = fmt::format(
    "("
        "basic_{{value}} > basic_{{negated_value}} AND "
        "("
            "basic_{{value}} > 1 OR "                 // Crutch: remediate most of expected status flapping
            "{basic_grades_are_no_longer_expected} "  // See https://st.yandex-team.ru/NMAPS-15476#62b1901dec457f57261a728b
        ") AND "
        "{last_expert_value} IS NULL OR "
        "{last_expert_value} = '{{value}}'"
    ")",

    "last_expert_value"_a = sql::col::LAST_EXPERT_VALUE,
    "basic_grades_are_no_longer_expected"_a = sql::col::LAST_GRADED_AT + " <= NOW() - '3 hours'::interval");

const std::string IS_OVERALL_CORRECT = fmt::format(
    OVERALL_VALUE_EQUALS,
    "value"_a = toString(Grade::Value::Correct),
    "negated_value"_a = toString(Grade::Value::Incorrect));

const std::string IS_OVERALL_INCORRECT = fmt::format(
    OVERALL_VALUE_EQUALS,
    "value"_a = toString(Grade::Value::Incorrect),
    "negated_value"_a = toString(Grade::Value::Correct));

} // namespace

std::string statusEquals(UnitStatus status)
{
    switch (status) {
    case UnitStatus::Correct:
        return IS_OVERALL_CORRECT;

    case UnitStatus::Incorrect:
        return IS_OVERALL_INCORRECT + " AND " + sql::col::FIXED_AT + " IS NULL";

    case UnitStatus::Fixed:
        return IS_OVERALL_INCORRECT + " AND " + sql::col::FIXED_AT + " IS NOT NULL";
    };
}

std::string fixable(TUid uid)
{
    return
        sql::col::ACTION_BY + " = " + std::to_string(uid) + " AND "
        "COALESCE(" + IS_OVERALL_INCORRECT + ", FALSE) AND " + sql::col::FIXED_AT + " IS NULL";
}

} // sqlClause

bool tryFix(pqxx::transaction_base& txn, TUid uid, TId unitId)
{
    const auto result = txn.exec(fmt::format(
        sqlClause::TRY_FIX_UNIT_FMT,
        "unit_id"_a = unitId,
        "fixable"_a = sqlClause::fixable(uid)));

    ASSERT(result.affected_rows() <= 1);
    return result.affected_rows();
}

} // maps::wiki::assessment::units
