#include <yandex/maps/wiki/common/default_config.h>

#include <library/cpp/resource/resource.h>
#include <set>

namespace maps::wiki::common {

namespace {

const std::string DEFAULT_CONFIG_PATH =
    "/etc/yandex/maps/wiki/services/services.xml";

const std::set<std::string> RESOURCE_ENVIRONMENTS = {
    "unstable", "testing", "load", "stable"
};

const std::map<std::string, std::string> ENVIRONMENT_TO_CANONICAL {
    { "development", "unstable" },
    { "stress",      "load" },
    { "production",  "stable" },
};

const std::string CORE_DB_XPATH =
    "/config/common/databases/database[@id=\"core\"]/*[self::host or self::write]";
const std::string CORE_DB_PASS = "pass";


class ResourceExtendedXmlDoc : public ExtendedXmlDoc
{
public:
    ResourceExtendedXmlDoc(
        const std::string& environment,
        const vault_boy::Context& context)
    {
        auto it = ENVIRONMENT_TO_CANONICAL.find(environment);
        const auto& env = it == ENVIRONMENT_TO_CANONICAL.end()
            ? environment
            : it->second;
        REQUIRE(RESOURCE_ENVIRONMENTS.count(env), "invalid environment: " << environment);

        auto filename = "services." + env + ".xml";
        do {
            auto xml = context.renderTemplate(NResource::Find(filename));
            filename = addDoc(filename, xml3::Doc::fromString(xml));
        }
        while (!filename.empty());
    }
};

} // namespace


std::unique_ptr<ExtendedXmlDoc> loadDefaultConfig()
{
    return std::make_unique<ExtendedXmlDoc>(DEFAULT_CONFIG_PATH);
}

std::unique_ptr<ExtendedXmlDoc> loadDefaultConfigFromResource(
    const std::string& environment,
    const vault_boy::Context& context)
{
    return std::make_unique<ResourceExtendedXmlDoc>(environment, context);
}

std::string getCoreDbPassword(const ExtendedXmlDoc& config)
{
    return config.getAttr<std::string>(CORE_DB_XPATH, CORE_DB_PASS);
}

} // namespace maps::wiki::common
