#pragma once

#include <maps/libs/common/include/exception.h>
#include <maps/wikimap/mapspro/libs/revision/include/yandex/maps/wiki/revision/objectrevision.h>
#include <maps/libs/ymapsdf/include/rd.h>

#include <boost/lexical_cast.hpp>

#include <map>
#include <string>
#include <type_traits>
#include <vector>

namespace maps::wiki::common {

class AttrValue {

public:
    AttrValue(std::string attributeId, std::string value);

    template<typename T> T as() const;
    template<typename T> T as(const T& defaultValue) const;

    explicit operator bool() const;
    template<typename T> operator T() const;

    bool equals(const AttrValue& other) const;

    bool equals(const char* other) const;

    bool equals(std::string_view other) const;

    bool equals(const std::string& other) const;

    template<typename T>
    bool equals(const T& other) const { return other == as<T>(); }

    bool operator==(const AttrValue& other) const;
    bool operator!=(const AttrValue& other) const;

    bool operator!() const;

    const std::string& value() const & { return value_; }
    std::string value() && { return std::move(value_); }

private:
    std::string attributeId_;
    std::string value_;
};

using AttrValues = std::vector<AttrValue>;

template<typename T>
T AttrValue::as() const
{
    try {
        using LexicalType = typename std::conditional<std::is_enum<T>::value, int, T>::type;

        return static_cast<T>(boost::lexical_cast<LexicalType>(value_));
    } catch (const std::bad_cast&) {
        throw RuntimeError() << "Bad attribute '" << attributeId_ << "': '" << value_ << "'";
    }
}

template<typename T>
T AttrValue::as(const T& defaultValue) const { return value_.empty() ? defaultValue : as<T>(); }

template<>
bool AttrValue::as() const;

template<>
ymapsdf::rd::Direction AttrValue::as() const;

template<>
std::string_view AttrValue::as() const = delete;

template<>
std::string AttrValue::as() const;

template<>
AttrValue::operator std::string_view() const = delete;

template<typename T>
AttrValue::operator T() const { return as<T>(); }

template<typename T>
bool operator==(const T& other, const AttrValue& value) { return value.equals(other); }

template<typename T>
bool operator==(const AttrValue& value, const T& other) { return value.equals(other); }

template<typename T>
bool operator!=(const T& other, const AttrValue& value) { return !value.equals(other); }

template<typename T>
bool operator!=(const AttrValue& value, const T& other) { return !value.equals(other); }


class AttrsWrap {

public:
    using StringMap = std::map<std::string, std::string>;
    using StringMultiMap = std::multimap<std::string, std::string>;

    explicit AttrsWrap(const StringMap& attributes);
    explicit AttrsWrap(StringMultiMap attributes);

    bool empty() const;

    AttrValue operator[](const char* attributeId) const;
    AttrValue operator[](std::string_view attributeId) const;
    AttrValue operator[](std::string attributeId) const;

    AttrValues range(const char* attributeId) const;
    AttrValues range(std::string_view attributeId) const;
    AttrValues range(const std::string& attributeId) const;

    static AttrsWrap extract(const revision::ObjectRevision& object);

private:
    StringMultiMap attributes_;
};

} // namespace maps::wiki::common
