#pragma once

#include <chrono>
#include <list>
#include <map>
#include <mutex>
#include <optional>

namespace maps::wiki::common {

template <typename Key, typename Value>
class LRUCache
{
public:
    LRUCache(size_t maxSize, std::chrono::milliseconds cacheTime)
        : maxSize_(maxSize)
        , cacheTime_(cacheTime)
    {}

    std::optional<Value> get(const Key& key)
    {
        auto it = key2valueData_.find(key);
        if (it == key2valueData_.end()) {
            return std::nullopt;
        }

        auto& valueData = it->second;
        lruList_.erase(valueData.lruIterator);
        if (now() < valueData.expires) {
            lruList_.push_front(LRUItem{it});
            valueData.lruIterator = lruList_.begin();
            return valueData.value;
        }
        key2valueData_.erase(it);
        return std::nullopt;
    }

    void put(const Key& key, Value value)
    {
        auto it = key2valueData_.find(key);
        if (it != key2valueData_.end()) {
            auto& valueData = it->second;
            lruList_.erase(valueData.lruIterator);
            lruList_.push_front(LRUItem{it});
            valueData.value = std::move(value);
            valueData.lruIterator = lruList_.begin();
            valueData.expires = now() + cacheTime_;
        } else {
            removeExpired();
            lruList_.push_front(LRUItem{key2valueData_.end()});
            auto pair = key2valueData_.emplace(
                key,
                ValueData{
                    std::move(value),
                    now() + cacheTime_,
                    lruList_.begin()});
            lruList_.front().valueIterator = pair.first;
        }
    }

    std::chrono::milliseconds cacheTime() const { return cacheTime_; }

private:
    static std::chrono::milliseconds now()
    {
        return std::chrono::duration_cast<std::chrono::milliseconds>(
            std::chrono::steady_clock::now().time_since_epoch());
    }

    void removeExpired()
    {
        if (key2valueData_.empty() || key2valueData_.size() < maxSize_) {
            return;
        }

        for (auto it = key2valueData_.begin(); it != key2valueData_.end(); ++it) {
            auto& valueData = it->second;
            if (valueData.expires <= now()) {
                lruList_.erase(valueData.lruIterator);
                key2valueData_.erase(it);
                return;
            }
        }

        // remove least used item
        key2valueData_.erase(lruList_.back().valueIterator);
        lruList_.pop_back();
    }

    struct LRUItem;
    using LRUList = std::list<LRUItem>;

    struct ValueData
    {
        Value value;
        std::chrono::milliseconds expires;
        typename LRUList::iterator lruIterator;
    };

    using Key2ValueData = std::map<Key, ValueData>;

    struct LRUItem
    {
        typename Key2ValueData::iterator valueIterator;
    };

    Key2ValueData key2valueData_;
    LRUList lruList_;
    size_t maxSize_;
    std::chrono::milliseconds cacheTime_;
};

template <typename Key, typename Value>
class LRUCacheThreadSafe
{
public:
    LRUCacheThreadSafe(size_t maxSize, std::chrono::milliseconds cacheTime)
        : cache_(maxSize, cacheTime)
    {}

    std::optional<Value> get(const Key& key)
    {
        std::lock_guard<std::mutex> lock(mutex_);
        return cache_.get(key);
    }

    void put(const Key& key, Value value)
    {
        std::lock_guard<std::mutex> lock(mutex_);
        cache_.put(key, std::move(value));
    }

private:
    std::mutex mutex_;
    LRUCache<Key, Value> cache_;
};

} // namespace maps::wiki::common
