#pragma once

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/retry.h>

#include <chrono>

namespace maps::wiki::common {

class RetryDurationException: public maps::Exception {
protected:
    RetryDurationException() = default;
};

class RetryDurationExpired: public RetryDurationException {
    using RetryDurationException::RetryDurationException;
};

class RetryDurationCancel: public RetryDurationException {
    using RetryDurationException::RetryDurationException;
};

class RetryDurationPolicy
{
public:
    RetryDurationPolicy();

    static void setDefaultDuration(std::chrono::seconds duration);
    static void setDefaultSleepTime(std::chrono::milliseconds sleepTime);

    auto duration() const { return duration_; }
    auto sleepTime() const { return sleepTime_; }

    size_t maxTries() const;

    RetryDurationPolicy& setDuration(std::chrono::seconds duration);
    RetryDurationPolicy& setSleepTime(std::chrono::milliseconds sleepTime);

    void checkDeadline(std::chrono::steady_clock::time_point deadline) const;

private:
    std::chrono::seconds duration_;
    std::chrono::milliseconds sleepTime_;
};

template<typename Function, typename Result = std::invoke_result_t<Function>>
Result retryDuration(Function function, const RetryDurationPolicy& policy = {})
{
    auto deadline = std::chrono::steady_clock::now() + policy.duration();

    auto timedFunction = [&] {
        policy.checkDeadline(deadline);
        return function();
    };

    auto retryPolicy = maps::common::RetryPolicy()
        .setInitialCooldown(policy.sleepTime())
        .setTryNumber(policy.maxTries());

    auto validate = [](const Expected<Result>& maybeResult) {
        return maybeResult.valid() ||
               maybeResult.template hasException<RetryDurationException>();
    };

    return maps::common::retry(timedFunction, retryPolicy, validate);
}

} // namespace maps::wiki::common
