#include <yandex/maps/wiki/common/revision_utils.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/profiletimer.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/commit.h>
#include <yandex/maps/wiki/revision/exception.h>
#include <yandex/maps/wiki/revision/filters.h>

#include <thread>

namespace maps::wiki::common {

namespace rf = revision::filters;

namespace {

constexpr auto TXN_WAIT_PERIOD = std::chrono::seconds(5);

bool isCommitInBranch(
    pqxx::transaction_base& txn,
    const revision::Branch& branch,
    revision::DBID commitId)
{
    auto commitIds = revision::Commit::loadIds(
        txn,
        rf::CommitAttr::id() == commitId && rf::CommitAttr::isVisible(branch));
    return !commitIds.empty();
}

} // namespace

pgpool3::TransactionHandle
getReadTransactionForCommit(
    pgpool3::Pool& pool,
    revision::DBID branchId,
    revision::DBID commitId,
    const LogFunc& logFunc)
{
    std::optional<revision::Branch> branch;
    try {
        auto slaveTxn = pool.slaveTransaction();
        branch = revision::BranchManager(*slaveTxn).load(branchId);

        if (isCommitInBranch(*slaveTxn, *branch, commitId)) {
            return slaveTxn;
        }
    } catch (const revision::BranchNotExistsException&) {
    }

    if (branch && branch->type() == revision::BranchType::Approved) {
        logFunc("waiting for commit " + std::to_string(commitId) + " to be approved");
    } else {
        auto masterTxn = pool.masterReadOnlyTransaction();
        if (!branch) {
            branch = revision::BranchManager(*masterTxn).load(branchId);
        }
        REQUIRE(isCommitInBranch(*masterTxn, *branch, commitId),
                "Commit id " << commitId
                << " does not exist in branch " << branchId << " yet");
        logFunc("waiting for replication of commit id " + std::to_string(commitId));
    }

    ProfileTimer timer;
    while (true) {
        std::this_thread::sleep_for(TXN_WAIT_PERIOD);

        auto slaveTxn = pool.slaveTransaction();
        if (isCommitInBranch(*slaveTxn, *branch, commitId)) {
            logFunc("waited for " + timer.getElapsedTime() + "s.");
            return slaveTxn;
        }
    }
}

} // namespace maps::wiki::common
