#include <yandex/maps/wiki/configs/editor/category_groups.h>

namespace maps {
namespace wiki {
namespace configs {
namespace editor {

class CategoryGroup::Impl
{
public:
    explicit Impl(const maps::xml3::Node& node)
        : id_(node.attr<std::string>("id"))
        , label_(node.firstElementChild("label").value<std::string>())
    {
        auto categoryNodes = node.nodes("categories/category", true);
        for (size_t j = 0; j < categoryNodes.size(); ++j) {
            categoryIds_.insert(categoryNodes[j].attr<std::string>("id"));
        }
        REQUIRE(!categoryIds_.empty(), "Empty category group, id: " << id_);
    }

    std::string id_;
    std::string label_;
    StringSet categoryIds_;
};

MOVABLE_PIMPL_DEFINITIONS(CategoryGroup)

CategoryGroup::CategoryGroup(const maps::xml3::Node& node)
    : impl_(new Impl{node})
{}

const std::string& CategoryGroup::id() const { return impl_->id_; }
const std::string& CategoryGroup::label() const { return impl_->label_; }
const StringSet& CategoryGroup::categoryIds() const { return impl_->categoryIds_; }

//========================================================

class CategoryGroups::Impl
{
public:
    explicit Impl(const maps::xml3::Node& node)
    {
        auto groupNodes = node.nodes("category-group");
        for (size_t i = 0; i < groupNodes.size(); ++i) {
            auto group = std::make_shared<CategoryGroup>(groupNodes[i]);
            groupsByIds_.insert({group->id(), group});
        }
    }

    GroupByIdMap groupsByIds_;
};

MOVABLE_PIMPL_DEFINITIONS(CategoryGroups)

CategoryGroups::CategoryGroups(const maps::xml3::Node& node)
    : impl_(new Impl{node})
{}

const StringSet&
CategoryGroups::categoryIdsByGroup(const std::string& groupId) const
{
    return groupById(groupId)->categoryIds();
}

StringSet
CategoryGroups::categoryIdsByGroups(const StringSet& groupIds) const
{
    StringSet categoryIds;
    for (const auto& groupId : groupIds) {
        const auto& groupPtr = groupById(groupId);
        const auto& groupCategories = groupPtr->categoryIds();
        categoryIds.insert(groupCategories.begin(), groupCategories.end());
    }
    return categoryIds;
}

GroupPtr
CategoryGroups::findGroupByCategoryId(const std::string& categoryId) const
{
    for (const auto& idAndGrp : impl_->groupsByIds_) {
        if (idAndGrp.second->categoryIds().count(categoryId)) {
            return idAndGrp.second;
        }
    }
    return {};
}

bool
CategoryGroups::isGroupExists(const std::string& groupId) const
{
    return impl_->groupsByIds_.count(groupId) > 0;
}

const GroupPtr&
CategoryGroups::groupById(const std::string& groupId) const
{
    auto it = impl_->groupsByIds_.find(groupId);
    REQUIRE(it != impl_->groupsByIds_.end(),
            "Category group " << groupId << " not found");
    return it->second;
}

const GroupByIdMap&
CategoryGroups::allGroups() const { return impl_->groupsByIds_; }

} // editor
} // configs
} // wiki
} // maps
