#include <yandex/maps/wiki/configs/editor/countries.h>

namespace maps {
namespace wiki {
namespace configs {
namespace editor {

Country::FcRangeSpeedLimits::FcRangeSpeedLimits(const maps::xml3::Node& node)
    : minFc_(node.attr<size_t>("min"))
    , maxFc_(node.attr<size_t>("max"))
    , inLocality_(node.attr<size_t>("in-locality"))
    , outLocality_(node.attr<size_t>("out-locality"))
{
}

Country::Country(const maps::xml3::Node& node)
    : isocode_(node.attr<std::string>("isocode"))
{
    auto rangeNodes = node.nodes("speed-limits/fc-range");
    for (size_t i = 0; i < rangeNodes.size(); ++i) {
        fcRangeSpeedLimits_.emplace_back(rangeNodes[i]);
    }
    auto prohibitedCategoriesNodes = node.nodes("prohibited-categories/category", true);
    for (size_t i = 0; i < prohibitedCategoriesNodes.size(); ++i) {
        prohibitedCategories_.insert(
            prohibitedCategoriesNodes[i].attr<std::string>("id"));
    }
    auto atrributeNodes = node.nodes("attributes/attribute", true);
    for (size_t attrIx = 0; attrIx < atrributeNodes.size(); ++attrIx) {
        const auto& attributeNode = atrributeNodes[attrIx];
        const auto attributeId = attributeNode.attr<std::string>("id");
        auto prohibitedValues = attributeNode.nodes("prohibited-values/value", true);
        for (size_t valueIx  = 0; valueIx  < prohibitedValues.size(); ++valueIx ) {
            prohibitedAttributeValues_.emplace(
                attributeId, prohibitedValues[valueIx].value<std::string>());
        }
    }
}

std::optional<Country::FcRangeSpeedLimits>
Country::fcRangeSpeedLimits(size_t fc) const
{
    for (const auto& range : fcRangeSpeedLimits_) {
        if (range.minFc() <= fc && fc <= range.maxFc()) {
            return range;
        }
    }
    return std::nullopt;
}

Countries::Countries(const maps::xml3::Node& node)
{
    maps::xml3::Node countryNode = node.firstElementChild("country");
    while (!countryNode.isNull()) {
        Country country(countryNode);
        countries_.insert({country.isocode(), country});
        countryNode = countryNode.nextElementSibling("country");
    }
    for (const auto& [isocode, country] : countries_) {
        for (const auto& categoryId : country.prohibitedCategories()) {
            categoryProhibitedIsocodes_[categoryId].insert(isocode);
            categoriesProhibitedSomewhere_.insert(categoryId);
        }
        for (const auto& attrValue : country.prohibitedAttributeValues()) {
            attrValueProhibitedIsocodes_[attrValue].insert(isocode);
        }
    }
}

bool Countries::hasCountryWithISOCODE(const std::string& isocode) const
{
    auto it = countries_.find(isocode);
    return it != countries_.end();
}

const Country&
Countries::findByISOCODE(const std::string& isocode) const
{
    auto it = countries_.find(isocode);
    REQUIRE(it != countries_.end(), "Failed to find country by the isocode " << isocode);
    return it->second;
}

Isocodes
Countries::isocodesWhereCategoryIsProhibited(const std::string& categoryId) const
{
    const auto it = categoryProhibitedIsocodes_.find(categoryId);
    return
        it == categoryProhibitedIsocodes_.end()
        ? Isocodes {}
        : it->second;
}

std::unordered_set<std::string>
Countries::categoriesProhibitedSomewhere() const
{
    return categoriesProhibitedSomewhere_;
}


Isocodes
Countries::isocodesWhereAttributeValueIsProhibited(const std::string& attributeId, const std::string& valueId) const
{
    const auto it = attrValueProhibitedIsocodes_.find(AttrValuePair{attributeId, valueId});
    return
        it == attrValueProhibitedIsocodes_.end()
        ? Isocodes {}
        : it->second;
}


} // editor
} // configs
} // wiki
} // maps
