#include <yandex/maps/wiki/configs/editor/externals.h>

using namespace std::string_literals;

namespace maps::wiki::configs::editor {

namespace {

const auto NAME = "name"s;
const auto LANG = "lang"s;
const auto BUSINESS_RUBRICS_NODE = "business-rubrics"s;
const auto RUBRICS_NODE = "Rubrics"s;
const auto RUBRIC_NODE = "Rubric"s;
const auto RUBRIC_ID_ATTR= "id"s;
const auto RUBRIC_DISP_CLASS_ATTR = "dispClass"s;
const auto FT_TYPE_ID_ATTR = "ftTypeId"s;

const auto GEOOBJECT_TEMPLATES_NODE = "geoobject-templates"s;
const std::vector<RubricId> EMPTY_RUBRICS;
const DispClass NO_DISP_CLASS = 0;
const FtTypeId NO_FT_TYPE_ID = 0;
} // namespace

Rubrics::Rubrics(const maps::xml3::Node& node)
{
    for (auto rubricNode = node.firstElementChild(RUBRIC_NODE);
        !rubricNode.isNull();
        rubricNode = rubricNode.nextElementSibling(RUBRIC_NODE))
    {
        const auto rubricId = rubricNode.attr<RubricId>(RUBRIC_ID_ATTR);
        const auto dispClass = rubricNode.attr<DispClass>(RUBRIC_DISP_CLASS_ATTR, NO_DISP_CLASS);
        const auto ftType = rubricNode.attr<FtTypeId>(FT_TYPE_ID_ATTR, NO_FT_TYPE_ID);
        Rubric rubric {
            rubricId,
            ftType ? std::optional<FtTypeId>(ftType) : std::nullopt,
            dispClass ? std::optional<DispClass>(dispClass) : std::nullopt,
            {}};
        for (auto nameNode = rubricNode.firstElementChild(NAME);
            !nameNode.isNull(); nameNode = nameNode.nextElementSibling(NAME))
        {
            rubric.names.emplace_back(
                common::LocalizedString {
                    nameNode.attr<std::string>(LANG),
                    nameNode.value<std::string>()});
        }
        rubricsById_.emplace(rubric.id, std::move(rubric));
    }
    for (const auto& [id, rubric] : rubricsById_) {
        for (const auto& name : rubric.names) {
            idxRubricsByName_[name].push_back(id);
        }
    }
}

const std::vector<RubricId>&
Rubrics::rubrics(const common::LocalizedString& name) const
{
    auto it = idxRubricsByName_.find(name);
    return it == idxRubricsByName_.end()
        ? EMPTY_RUBRICS
        : it->second;
}

const Rubric*
Rubrics::rubric(RubricId rubricId) const
{
    auto it = rubricsById_.find(rubricId);
    return it == rubricsById_.end()
        ? nullptr
        : &it->second;
}

std::optional<FtTypeId>
Rubrics::defaultFtType(RubricId rubricId) const
{
    auto it = rubricsById_.find(rubricId);
    return it == rubricsById_.end()
        ? std::nullopt
        : it->second.ftType;
}

std::optional<DispClass>
Rubrics::defaultDispClass(RubricId rubricId) const
{
    auto it = rubricsById_.find(rubricId);
    return it == rubricsById_.end()
        ? std::nullopt
        : it->second.dispClass;
}

Externals::Externals(const maps::xml3::Node& node)
    : rubrics_(node.node(BUSINESS_RUBRICS_NODE).firstElementChild(RUBRICS_NODE))
    , geoObjectTemplates_(node.node(GEOOBJECT_TEMPLATES_NODE))
{
}

} // namespace maps::wiki::configs::editor

