#pragma once

#include <optional>
#include <list>
#include <map>

namespace maps {
namespace wiki {

template <class T, typename KeyType = T>
class UniqueList
{
public:
    typedef std::list<T> StorageType;
    typedef typename StorageType::iterator iterator;
    typedef typename StorageType::const_iterator const_iterator;
    typedef typename StorageType::value_type value_type;
    typedef typename StorageType::reference reference;
    typedef typename StorageType::const_reference const_reference;

    UniqueList() = default;

    UniqueList(const UniqueList& other)
    {
        for(const auto& o: other){
            push_back(o);
        }
    }

    UniqueList& operator =(const UniqueList& other)
    {
        UniqueList tmp(other);
        tmp.swap(*this);
        return *this;
    }

    UniqueList(UniqueList&&) = default;
    UniqueList& operator =(UniqueList&&) = default;

    bool operator ==(const UniqueList& other) const
    {
        if (size() != other.size()) {
            return false;
        }
        for (const auto& element : data_) {
            if (!other.contains(element)) {
                return false;
            }
        }
        return true;
    }

    bool contains(const T& element) const
    {
        return index_.contains(KeyType(element));
    }

    const_iterator findByKey(const KeyType& key) const
    {
        auto it = index_.find(key);
        return (it == index_.end())
            ? end()
            : it->second;
    }
    iterator findByKey(const KeyType& key)
    {
        auto it = index_.find(key);
        return (it == index_.end())
            ? end()
            : it->second;
    }
    const_iterator find(const T& o) const
    {
        auto it = index_.find(KeyType(o));
        return (it == index_.end())
            ? end()
            : it->second;
    }
    iterator find(const T& o)
    {
        auto it = index_.find(KeyType(o));
        return (it == index_.end())
            ? end()
            : it->second;
    }
    void erase(iterator it)
    {
        auto itIndex = index_.find(KeyType(*it));
        if (itIndex != index_.end()) {
            data_.erase(it);
            index_.erase(itIndex);
        }
    }

    std::optional<T> extractByKey(const KeyType& key)
    {
        auto it = index_.find(key);
        if (it == index_.end()) {
            return std::nullopt;
        }
        auto dataIt = it->second;
        ASSERT(dataIt != data_.end());
        std::optional<T> result(std::move(*dataIt));
        data_.erase(dataIt);
        index_.erase(it);
        return result;
    }

    void removeByKey(const KeyType& key)
    {
        auto it = index_.find(key);
        if (it != index_.end()) {
            erase(it->second);
        }
    }

    void remove(const T& o)
    {
        removeByKey(KeyType(o));
    }

    template <class Predicate> void removeIf(Predicate pred)
    {
        for (auto itIndex = index_.begin(); itIndex != index_.end();) {
            if (pred(*(itIndex->second))) {
                data_.erase(itIndex->second);
                itIndex = index_.erase(itIndex);
            } else {
                 ++itIndex;
            }
        }
    }

    template<typename U>
    void push_back(U&& o)
    {
        KeyType key(o);
        push_back(std::forward<U>(o), key); //OK
    }

    template<typename U>
    void push_back(U&& o, const KeyType& key)
    {
        auto mapResult = index_.insert({key, data_.end()});
        if (mapResult.second) {
            mapResult.first->second = data_.insert(data_.end(), std::forward<U>(o));
        }
    }

    const_iterator begin() const { return data_.begin(); }
    const_iterator end() const { return data_.end(); }
    iterator begin() { return data_.begin(); }
    iterator end() { return data_.end(); }

    const_reference front() const { return data_.front(); }
    const_reference back() const { return data_.back(); }
    reference front() { return data_.front(); }
    reference back() { return data_.back(); }

    size_t size() const { return data_.size(); }
    size_t empty() const { return data_.empty(); }

    void swap(UniqueList& other)
    {
        index_.swap(other.index_);
        data_.swap(other.data_);
    }

    bool indexEqual(const UniqueList& other) const
    {
        if (index_.size() != other.index_.size()) {
            return false;
        }
        for(auto itThis = index_.begin(), itOther = other.index_.begin();
                itThis != index_.end();
                ++itThis, ++itOther) {
            if (!(itThis->first == itOther->first)) {
                return false;
            }
        }
        return true;
    }

    void clear()
    {
        index_.clear();
        data_.clear();
    }

private:
    std::map<KeyType, iterator> index_;
    StorageType data_;
};

}//wiki
}//maps
