#include <maps/wikimap/mapspro/libs/controller/include/async_tasks_support.h>

#include <yandex/maps/wiki/threadutils/thread_observer.h>
#include <maps/wikimap/mapspro/libs/revision_meta/include/branch_lock_ids.h>
#include <yandex/maps/wiki/common/extended_xml_doc.h>
#include <yandex/maps/wiki/common/pgpool3_helpers.h>
#include <yandex/maps/wiki/threadutils/threadpool.h>
#include <maps/wikimap/mapspro/libs/taskutils/include/taskutils.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <maps/libs/log8/include/log8.h>

#include <chrono>

namespace maps::wiki::controller {

namespace {

const std::string TASKUTILS_TOKEN_SECRET_WORD = "barbambea";
const size_t TASKUTILS_DEFAULT_DEADLINE_TIMEOUT = 30 * 60; // 30 minutes
const size_t TASKUTILS_DEFAULT_WAIT_RESPONSE_TIMEOUT = 3; // 3 seconds
const size_t TASKUTILS_DEFAULT_KEEP_EXPIRED_TIME = 14 * 24 * 60 * 60; // 2 weeks

const std::string REMOVER_NAME = "taskutils expired data remover";
const std::chrono::hours REMOVER_UPDATE_DELAY(1);

struct RemoverAgent
{
    pgpool3::Pool& poolCore;
    std::shared_ptr<taskutils::TaskManager> manager;
    size_t keepExpiredTime;

    size_t run() const;
};

size_t RemoverAgent::run() const
{
    auto work = poolCore.masterWriteableTransaction();
    auto branch = revision::BranchManager(*work).loadTrunk();

    if (branch.state() != revision::BranchState::Normal) {
        INFO() << REMOVER_NAME
               << " skipped, branch not normal, state: " << branch.state();
        return 0;
    }

    if (!branch.tryLock(
            *work,
            revision_meta::REMOVE_EXPIRED_ASYNC_TASKS_LOCK_ID,
            revision::Branch::LockType::Exclusive)) {
        INFO() << REMOVER_NAME << " skipped, branch already locked";
        return 0;
    }

    return manager->removeExpired(*work, keepExpiredTime);
}

} // namespace


class AsyncTasksSupport::Impl
{
public:
    Impl(
        pgpool3::Pool& poolCore,
        const common::ExtendedXmlDoc& doc,
        const std::string& configSectionPath);

    ~Impl();

    void shutdown();
    void terminateTasks();

    pgpool3::Pool& poolCore_;
    std::unique_ptr<AsyncTasksParams> params_;
    std::shared_ptr<taskutils::TaskManager> manager_;
    std::unique_ptr<ThreadPool> threadPool_;

private:
    class Remover: public ThreadObserver<Remover>
    {
    public:
        explicit Remover(RemoverAgent agent)
            : ThreadObserver<Remover>(REMOVER_UPDATE_DELAY)
            , agent_(std::move(agent))
        {
            start();
        }

    private:
        friend class ThreadObserver<Remover>;
        constexpr const std::string& name() const noexcept { return REMOVER_NAME; }
        void onStart() { INFO() << name() << " started"; }
        void onStop() { INFO() << name() << " stopped"; }

        void doWork()
        try {
            auto removedSize = agent_.run();
            if (removedSize) {
                INFO() << name() << " : " << removedSize << " records processed";
            }
        }
        catch (const maps::Exception& ex) {
            ERROR() << name() << " failed: " << ex;
        }
        catch (const std::exception& ex) {
            ERROR() << name() << " failed: " << ex.what();
        }

        RemoverAgent agent_;
    };

    std::unique_ptr<Remover> remover_;
};

AsyncTasksSupport::Impl::Impl(
        pgpool3::Pool& poolCore,
        const common::ExtendedXmlDoc& doc,
        const std::string& configSectionPath)
    : poolCore_(poolCore)
{
    const size_t defaultThreads = poolCore_.state().constants.masterMaxSize / 3 + 1;
    AsyncTasksParams params = {
        doc.get<size_t>(
            configSectionPath + "timeouts/waitResponseTimeout",
            TASKUTILS_DEFAULT_WAIT_RESPONSE_TIMEOUT),
        doc.get<size_t>(
            configSectionPath + "timeouts/methodTimeout",
            TASKUTILS_DEFAULT_DEADLINE_TIMEOUT)
    };
    params_ = std::make_unique<AsyncTasksParams>(params);

    auto keepExpiredTime =
        doc.get<size_t>(
            configSectionPath + "timeouts/keepExpiredTime",
            TASKUTILS_DEFAULT_KEEP_EXPIRED_TIME);

    auto threads = doc.get<size_t>(
        configSectionPath + "threads", defaultThreads);

    auto tokenSecretWord = doc.get<std::string>(
            configSectionPath + "tokenSecretWord", TASKUTILS_TOKEN_SECRET_WORD);

    manager_ = std::make_shared<taskutils::TaskManager>(tokenSecretWord);
    threadPool_ = std::make_unique<ThreadPool>(threads);

    if (keepExpiredTime) {
        remover_ = std::make_unique<Remover>(
            RemoverAgent{poolCore_, manager_, keepExpiredTime}
        );
    }
}

AsyncTasksSupport::Impl::~Impl() = default;

void
AsyncTasksSupport::Impl::shutdown()
{
    threadPool_->shutdown();
    remover_.reset();
}

void
AsyncTasksSupport::Impl::terminateTasks()
{
    try {
        auto work = poolCore_.masterWriteableTransaction();
        manager_->terminateAll(*work);
    } catch (const std::exception& ex) {
        ERROR() << "Termination async tasks failed: " << ex.what();
    }
}


AsyncTasksSupport::AsyncTasksSupport(
        pgpool3::Pool& poolCore,
        const common::ExtendedXmlDoc& doc,
        const std::string& configSectionPath)
    : impl_(new Impl(poolCore, doc, configSectionPath))
{}

AsyncTasksSupport::~AsyncTasksSupport() = default;

const taskutils::TaskManager&
AsyncTasksSupport::manager() const
{
    return *impl_->manager_;
}

const AsyncTasksParams&
AsyncTasksSupport::params() const
{
    return *impl_->params_;
}

ThreadPool&
AsyncTasksSupport::threadPool() const
{
    return *impl_->threadPool_;
}

pgpool3::Pool&
AsyncTasksSupport::corePool() const
{
    return impl_->poolCore_;
}

void
AsyncTasksSupport::shutdown()
{
    impl_->shutdown();
}

void
AsyncTasksSupport::terminateTasks()
{
    impl_->terminateTasks();
}

} // namespace maps::wiki::controller
