#include <maps/wikimap/mapspro/libs/controller/include/asyncbasecontroller.h>
#include <maps/wikimap/mapspro/libs/controller/include/async_control_support.h>
#include <maps/wikimap/mapspro/libs/controller/include/asynctaskcontext.h>
#include <maps/wikimap/mapspro/libs/controller/include/exception.h>

#include "json_strings.h"

#include <yandex/maps/wiki/threadutils/threadpool.h>

#include <maps/libs/pgpool/include/pgpool3.h>
#include <maps/libs/log8/include/log8.h>
#include <mutex>
#include <condition_variable>

namespace maps::wiki::controller {

const std::string TASK_SERVICE_NAME = "mapspro";

class AsyncTaskContextProxy
{
public:
    AsyncTaskContextProxy(
        const AsyncTasksSupport& asyncTasksSupport,
        const std::shared_ptr<AsyncTaskContext>& ctx)
        : asyncTasksSupport_(asyncTasksSupport)
        , ctx_(ctx)
        , data_(std::make_shared<SharedData>())
    {}

    void operator ()()
    {
        data_->setResult(ctx_->run(asyncTasksSupport_));
    }

    std::string waitAsyncResult(size_t timeout) const
    {
        return data_->waitAsyncResult(timeout).fetch();
    }

private:
    class SharedData {
    public:
        SharedData() {}

        void setResult(AsyncTaskContext::Result&& result)
        {
            std::unique_lock<std::mutex> lock(mutex_);
            result_ = std::move(result);
            cond_.notify_one();
        }

        AsyncTaskContext::Result waitAsyncResult(size_t timeout)
        {
            std::unique_lock<std::mutex> lock(mutex_);
            if (result_.empty()) {
                cond_.wait_for(lock, std::chrono::seconds(timeout));
            }
            return result_;
        }

    private:
        std::mutex mutex_;
        std::condition_variable cond_;
        AsyncTaskContext::Result result_;
    };
    const AsyncTasksSupport& asyncTasksSupport_;
    std::shared_ptr<AsyncTaskContext> ctx_;
    std::shared_ptr<SharedData> data_;
};

AsyncBaseController::AsyncBaseController(
    const std::string& profileWhere,
    WaitPolicy waitPolicy,
    const AsyncTasksSupport& asyncTasksSupport)
    : BaseController<AsyncBaseController>(profileWhere)
    , waitPolicy_(waitPolicy)
    , asyncTasksSupport_(asyncTasksSupport)
{
}

void
AsyncBaseController::execTaskContext(const std::shared_ptr<AsyncTaskContext>& ctx)
{
    AsyncTaskContextProxy taskContextProxy(asyncTasksSupport_, ctx);
    REQUIRE(asyncTasksSupport_.threadPool().push(taskContextProxy),
        "Thread pool already stopped");

    if (waitPolicy_ == WaitPolicy::Timeout) {
        auto timeout = asyncTasksSupport_.params().waitResponseTimeout;
        if (timeout) {
            result_->result = taskContextProxy.waitAsyncResult(timeout);
        }
    }
}

std::optional<taskutils::Task>
AsyncBaseController::createTask(
    taskutils::TUid uid,
    const AsyncTaskControlData& controlData)
{
    auto work = asyncTasksSupport_.corePool().masterWriteableTransaction();
    if (!controlData.recoveryPatternJson.empty()) {
        auto existingTasksTokens =
            asyncTasksSupport_.manager().findTasksTokens(
                *work, controlData.recoveryPatternJson, uid,
                taskutils::TaskManager::FindPolicy::ActiveAndDone);
            if (existingTasksTokens.size() == 1) {
                result_->tokenStr = existingTasksTokens.front().str();
                result_->taskName = taskName();
                return std::nullopt;
            }
            REQUIRE(existingTasksTokens.empty(),
                "Non unique async task match for user: "
                    << uid << " and pattern: " << controlData.recoveryPatternJson);

    }
    if (!controlData.conflictPatternJson.empty()) {
        auto existingTasksTokens =
            asyncTasksSupport_.manager().findTasksTokens(
                *work, controlData.conflictPatternJson, uid,
                taskutils::TaskManager::FindPolicy::Active);
            DEBUG() << "Conflict pattern:" << controlData.conflictPatternJson;
            REQUIRE(existingTasksTokens.empty(),
                ConflictingOperation() <<
                "User: " << uid << " attempt to create conflicting task: " << taskName()
                << " conflict pattern: " << controlData.conflictPatternJson);
    }
    const auto task = asyncTasksSupport_.manager().create(
        *work,
        taskutils::TaskInfo(
            TASK_SERVICE_NAME,
            taskName(),
            controlData.requestData,
            controlData.metadataJson),
        uid,
        asyncTasksSupport_.params().methodTimeout);

    result_->tokenStr = task.token().str();
    result_->taskName = taskName();
    return task;
}

std::string
ResultType<controller::AsyncBaseController>::toJson() const
{
    if (!result.empty()) {
        return result;
    }
    json::Builder builder;
    builder << [&](json::ObjectBuilder objectBuilder) {
        objectBuilder[json_strings::STR_TASK] << [&](json::ObjectBuilder taskBuilder) {
            taskBuilder[json_strings::STR_NAME] = taskName;
            taskBuilder[json_strings::STR_ID] = tokenStr;
            taskBuilder[json_strings::STR_STATUS] = toString(taskutils::TaskStatus::Created);
        };
    };
    return builder.str();
}

} // namespace maps::wiki::controller
