#pragma once

#include <maps/wikimap/mapspro/libs/controller/include/profile.h>
#include <maps/wikimap/mapspro/libs/taskutils/include/common.h>
#include <maps/libs/common/include/exception.h>
#include <string>
#include <sstream>

namespace maps::wiki::controller {

template <typename Controller> struct ResultType;
/**
* Base handler for various servant tasks
*/
template <typename DerivedController>
class BaseController
{
public:
    typedef ResultType<DerivedController> DerivedControllerResultType;
    typedef std::unique_ptr<DerivedControllerResultType> DerivedControllerResultTypePtr;

    BaseController() = delete;
    BaseController(const BaseController&) = delete;
    BaseController(BaseController&&) = delete;

    BaseController(const std::string& name, taskutils::TaskID asyncTaskID = 0)
        : result_(std::make_unique<DerivedControllerResultType>())
        , profile_(name)
        , asyncTaskID_(asyncTaskID)
    {
    }

    virtual ~BaseController() {}

    /**
    * Perform task and return result
    */
    DerivedControllerResultTypePtr operator ()()
    {
        ProfileUpdater profileUpdate(profile_, *this);
        control();
        REQUIRE(result_, "Result is not calculated/updated yet");
        profile_.success();
        return std::move(result_);
    }

    //! Export stat interface
    Profile& profile()
    { return profile_; }

    taskutils::TaskID asyncTaskID() const { return asyncTaskID_; }

    /**
    * This method implementation shall print controller
    * call details
    */
    virtual std::string printRequest() const = 0;

protected:
    /**
    * Override to implement controller specific actions
    */
    virtual void control() = 0;

protected:
    class ProfileUpdater
    {
    public:
        ProfileUpdater(Profile& profile, BaseController& controller)
            : profile_(profile)
            , controller_(controller)
        {}

        ~ProfileUpdater()
        {
            profile_.setCallDetails(controller_.printRequest(), controller_.asyncTaskID());
        }

        Profile& profile_;
        BaseController& controller_;
    };
    DerivedControllerResultTypePtr result_;

private:
    Profile profile_;
    taskutils::TaskID asyncTaskID_;
};

} // namespace maps::wiki::controller
