#include "ad.h"
#include "magic_string.h"
#include "utils.h"

#include <yandex/maps/wiki/diffalert/object.h>
#include <yandex/maps/wiki/diffalert/snapshot.h>

#include <yandex/maps/wiki/common/misc.h>

namespace maps::wiki::diffalert {
namespace {

enum class ReportAll
{
    Yes,
    No
};

bool isFormalLocality(const Object& object)
{
    const auto levelKind = getLevelKind(object);
    const auto informal = object.attr(attr::INFORMAL).as<bool>();
    return LevelKind::Locality == levelKind && !informal;
}

void checkAdChanges(
        const DiffContext& diff, MessageReporter& messages, CalcSortPriority calcSortPriority)
{
    if (diff.categoryId() != cat::AD) {
        return;
    }

    enum class Importance { Minor, MinorPopulation1E4, MinorPopulation1E5, Town, Major };

    auto getImportance = [](const Object& object) {
        const auto levelKind = getLevelKind(object);
        const auto population = object.attr(attr::POPULATION).as<size_t>(0);
        if (LevelKind::Country <= levelKind && levelKind <= LevelKind::Area) {
            return Importance::Major;
        } else if (LevelKind::Locality == levelKind && object.attr(attr::TOWN).as<bool>()) {
            return Importance::Town;
        } else if (population >= 100000) {
            return Importance::MinorPopulation1E5;
        } else if (population >= 10000) {
            return Importance::MinorPopulation1E4;
        }
        return Importance::Minor;
    };

    auto reportForImportance = [&](
            Importance importance,
            const std::string& messageSuffix,
            Message::Scope scope,
            double sortPriority,
            ReportAll reportAll = ReportAll::No)
    {
        if (Importance::Major == importance) {
            messages.report({0, 0, sortPriority}, "major-ad-" + messageSuffix, scope);
        } else if (Importance::Town == importance) {
            messages.report({0, 1, sortPriority}, "ad-town-" + messageSuffix, scope);
        } else if (Importance::MinorPopulation1E5 == importance) {
            messages.report(
                {1, 4, sortPriority}, "minor-population-1e5-ad-" + messageSuffix, scope);
        } else if (
            Importance::MinorPopulation1E4 == importance ||
            reportAll == ReportAll::Yes)
        {
            messages.report(
                {3, 1, sortPriority}, "minor-population-1e4-ad-" + messageSuffix, scope);
        }
    };

    if (!diff.oldObject()) {
        const auto importance = getImportance(*diff.newObject());
        reportForImportance(importance, "created",
            Message::Scope::WholeObject, 0.0, ReportAll::Yes);
    } else if (!diff.newObject()) {
        const auto importance = getImportance(*diff.oldObject());
        if (importance >= Importance::Town) {
            reportForImportance(importance, "deleted",
                Message::Scope::WholeObject, 0.0);
        }
    } else {
        const auto importance = std::max(
            getImportance(*diff.oldObject()),
            getImportance(*diff.newObject())
        );
        if (importance > Importance::Minor) {
            if (diff.geomChanged()) {
                double sortPriority =
                    calcSortPriority == CalcSortPriority::Yes ? -symDiffArea(diff) : 0.0;
                reportForImportance(importance, "geom-changed",
                                    Message::Scope::GeomDiff,
                                    sortPriority);
            }
        }
        if (diff.attrsChanged() || diff.tableAttrsChanged()) {
            auto formalLocality = isAny(diff, isFormalLocality);
            if (importance < Importance::Town && formalLocality) {
                messages.report({1, 1, 0}, "ad-locality-attrs-changed",
                    Message::Scope::WholeObject);
            } else if (importance > Importance::Minor) {
                reportForImportance(importance, "attrs-changed",
                    Message::Scope::WholeObject, 0.0);
            }

            if (importance < Importance::Town && !formalLocality && namesChanged(diff)) {
                messages.report({2, 1}, "minor-ad-names-changed");
            }
        }
    }
}

} // namespace

void checkAdChangesEditor(const DiffContext& d, MessageReporter& mr)
{
    checkAdChanges(d, mr, CalcSortPriority::No);
}
void checkAdChangesLongtask(const DiffContext& d, MessageReporter& mr)
{
    checkAdChanges(d, mr, CalcSortPriority::Yes);
}

void checkCapitalAd(const DiffContext& d, MessageReporter& messages)
{
    if (!(d.categoryId() == cat::AD && d.attrsChanged())) {
        return;
    }
    auto capitalAttrValue = [](OptionalObject object) {
        if (object) {
            return object->attr(attr::CAPITAL).as<int>(0);
        }
        return 0;
    };
    if (capitalAttrValue(d.oldObject()) != capitalAttrValue(d.newObject())) {
        messages.report({0, 1}, "ad-capital-attr-changed");
    }
}

void checkAdHierarchyChange(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::AD && diff.relationsChanged() && !diff.stateChanged())) {
        return;
    }

    auto getParentId = [&diff](const Relations& relations) -> TId
    {
        for (const auto& relation: relations) {
            if (diff.objectId() == relation.slaveId && relation.role == role::CHILD) {
                return relation.masterId;
            }
        }
        return 0;
    };

    auto exclusionChanged = [&diff](const Relations& relations)
    {
        for (const auto& relation: relations) {
            if (diff.objectId() == relation.masterId && relation.role == role::EXCLUSION) {
                return true;
            }
        }
        return false;
    };

    auto getParentLevelKind = [](Snapshot& snapshot, const TId parentId)
    {
        const auto result = snapshot.objectsByIds({parentId});
        ASSERT(result.size() == 1);
        return getLevelKind(*result.front());
    };


    const auto oldParentId = getParentId(diff.relationsDeleted());
    const auto newParentId = getParentId(diff.relationsAdded());

    if (oldParentId != newParentId) {
        messages.report({0, 2}, "ad-hierarchy-parent-changed");

        if (oldParentId && newParentId) {
            const auto oldParentLevelKind = getParentLevelKind(diff.oldSnapshot(), oldParentId);
            const auto newParentLevelKind = getParentLevelKind(diff.newSnapshot(), newParentId);

            if (std::abs(oldParentLevelKind - newParentLevelKind) > 1) {
                messages.report({2, 1}, "ad-hierarchy-parent-level-kind-changed");
            }
        }
    }
    if (exclusionChanged(diff.relationsDeleted()) || exclusionChanged(diff.relationsAdded())) {
        messages.report({0, 2}, "ad-hierarchy-exclusion-changed");
    }
}

void checkAdRecognitionChange(const DiffContext& d, MessageReporter& messages)
{
    if (!((common::isIn(d.categoryId(), { cat::AD, cat::AD_SUBST }))
          && d.oldObject() && d.newObject()
          && d.attrsChanged())) {
        return;
    }

    if (d.oldObject()->attr(attr::RECOGNITION)
          != d.newObject()->attr(attr::RECOGNITION)) {
        messages.report({0, 2}, "ad-recognition-changed");
    }
}

void checkAdDispClass(const DiffContext& diff, MessageReporter& messages)
{
    if (diff.categoryId() != cat::AD) {
        return;
    }

    if (diff.newObject()) {
        const auto newDispClass = getDispClass(*diff.newObject());
        if (newDispClass == DISP_CLASS_IGNORED && (
                !diff.oldObject()
                    || getDispClass(*diff.oldObject()) != DISP_CLASS_IGNORED))
        {
            messages.report({2, 3}, "ad-disp-class-set-to-ignored");
        }
    }

    if (diff.oldObject() && diff.newObject()) {
        const auto oldDispClass = getDispClass(*diff.oldObject());
        const auto newDispClass = getDispClass(*diff.newObject());
        if (newDispClass == DISP_CLASS_IGNORED && namesCreated(diff)) {
            messages.report({2, 3}, "ad-disp-class-ignored-names-created");
        }
        if (std::abs(oldDispClass - newDispClass) >= 2) {
            messages.report({2, 3}, "ad-disp-class-changed");
        }
    }
}

void checkAdLevelKind(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::AD && diff.attrsChanged())) {
        return;
    }

    if (diff.newObject()) {
        const auto town = diff.newObject()->attr(attr::TOWN).as<bool>();
        const auto municipality = diff.newObject()->attr(attr::MUNICIPALITY).as<bool>();
        const auto levelKind = getLevelKind(*diff.newObject());

        if (town && (LevelKind::Locality != levelKind)) {
            messages.report({1, 3}, "ad-wrong-town-level-kind");
        }

        if (municipality && (LevelKind::Locality != levelKind)) {
            messages.report({1, 2}, "ad-wrong-municipality-level-kind");
        }

        if (levelKind < LevelKind::Locality
            || (levelKind == LevelKind::Locality && town)) {
            const auto informal = diff.newObject()->attr(attr::INFORMAL).as<bool>();
            if (informal) {
                messages.report({1, 2}, "ad-wrong-informal-level-kind");
            }
        }
    }

    if (diff.oldObject() && diff.newObject()) {
        const auto oldLevelKind = getLevelKind(*diff.oldObject());
        const auto newLevelKind = getLevelKind(*diff.newObject());
        if ((LevelKind::Block == oldLevelKind && LevelKind::Locality >= newLevelKind)
                || (LevelKind::Other == oldLevelKind
                        && LevelKind::Locality >= newLevelKind)
                || (LevelKind::District == oldLevelKind
                        && LevelKind::District != newLevelKind))
        {
            messages.report({2, 2}, "ad-level-kind-suspicious-change");
        }

        const auto oldTown = diff.oldObject()->attr(attr::TOWN).as<bool>();
        const auto newTown = diff.newObject()->attr(attr::TOWN).as<bool>();
        if ((LevelKind::Locality == oldLevelKind || LevelKind::Locality == newLevelKind)
            && oldTown != newTown) {
             messages.report({0, 1}, "ad-town-changed");
        }

        const auto oldMunicipality = diff.oldObject()->attr(attr::MUNICIPALITY).as<bool>();
        const auto newMunicipality = diff.newObject()->attr(attr::MUNICIPALITY).as<bool>();
        if (oldMunicipality && !newMunicipality) {
             messages.report({1, 2}, "ad-municipality-changed");
        }

        const auto oldInformal = diff.oldObject()->attr(attr::INFORMAL).as<bool>();
        const auto newInformal = diff.newObject()->attr(attr::INFORMAL).as<bool>();
        if (oldInformal && !newInformal) {
             messages.report({1, 2}, "ad-informal-changed");
        }
    }
}

void checkAdBigAreaCreated(const DiffContext& d, MessageReporter& messages)
{
    if (!(d.categoryId() == cat::AD
          && !d.oldObject() && d.newObject())) {
        return;
    }

    const double AREA_THRESHOLD = 100*1000*1000; // square meters
    auto area = contourObjectArea(d, SnapshotTime::New);
    if (area.exterior >= AREA_THRESHOLD) {
        double sortPriority = -area.exterior;
        messages.report({3, 2, sortPriority}, "ad-big-area-created", Message::Scope::GeomDiff);
    }
}

void checkAdPopulationChange(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::AD && diff.oldObject() && diff.newObject())) {
        return;
    }

    const Object& oldObject = *diff.oldObject();
    const Object& newObject = *diff.newObject();

    if (!oldObject.attr(attr::POPULATION) && !newObject.attr(attr::POPULATION)) {
        return;
    }

    if (!oldObject.attr(attr::POPULATION) && newObject.attr(attr::POPULATION)) {
        messages.report({1, 1}, "ad-population-is-set");
        return;
    }

    if (oldObject.attr(attr::POPULATION) && !newObject.attr(attr::POPULATION)) {
        messages.report({1, 1}, "ad-population-is-unset");
        return;
    }

    const auto oldPopulation = oldObject.attr(attr::POPULATION).as<size_t>();
    const auto newPopulation = newObject.attr(attr::POPULATION).as<size_t>();

    if (oldPopulation == newPopulation) {
        return;
    }

    if (oldPopulation == 0 || newPopulation / oldPopulation >= 2) {
        messages.report({1, 1}, "ad-population-significant-increased");
    }

    if (newPopulation == 0 || oldPopulation / newPopulation >= 2) {
        messages.report({1, 1}, "ad-population-significant-decreased");
    }
}

} // namespace maps::wiki::diffalert
