#include "bld.h"
#include "utils.h"
#include "magic_string.h"

#include "../message_reporter.h"

#include <yandex/maps/wiki/diffalert/snapshot.h>
#include <yandex/maps/wiki/diffalert/diff_context.h>

#include <yandex/maps/wiki/common/misc.h>

namespace maps {
namespace wiki {
namespace diffalert {

namespace {

enum class Condition {
    Normal = 0,
    UnderConstruction = 1,
    Deserted = 2,
    Demolished = 3
};

Condition getCondition(const Object& object)
{
    return static_cast<Condition>(object.attr(attr::COND).as<int>(0));
}

} // namespace

void checkBldWithModel3D(const DiffContext& d, MessageReporter& messages)
{
    if (d.categoryId() != cat::BLD) {
        return;
    }

    auto hasRelsToModel3D = [&](const Relations& rels)
    {
        for (const auto& rel : rels) {
            if (rel.slaveId == d.objectId() && rel.role == role::ASSOCIATED_WITH) {
                return true;
            }
        }
        return false;
    };
    auto modelAdded = hasRelsToModel3D(d.relationsAdded());
    auto modelDeleted = hasRelsToModel3D(d.relationsDeleted());

    if (modelDeleted && !modelAdded) {
        messages.report({1, 4}, "bld-model3d-deleted");
        return;
    } else if (modelAdded && !modelDeleted) {
        messages.report({1, 4}, "bld-model3d-added");
    } else if (modelAdded && modelDeleted) {
        messages.report({1, 4}, "bld-model3d-changed");
    } else if (d.oldObject() && d.newObject()
               && !hasRelsToModel3D(d.newObject()->loadMasterRelations())) {
        // ordinary building without 3D model
        return;
    }

    if (d.oldObject() && d.newObject()) {
        if (d.attrsChanged()) {
            messages.report({1, 4}, "bld-with-model3d-attrs-changed");
        }
        if (d.geomChanged()) {
            messages.report({1, 4}, "bld-with-model3d-geom-changed",
                            Message::Scope::GeomDiff);
        }
    }
}

void checkBigBld(const DiffContext& d, MessageReporter& messages)
{
    auto isBigBld = [](const Object& bld)
    {
        double ratio = mercatorDistanceRatio(bld.geom());
        return bld.geom()->getArea() * ratio * ratio >= 50000
            || bld.attr(attr::HEIGHT).as<int>(0) >= 70;
    };
    if (!(d.categoryId() == cat::BLD
          && isAny(d, isBigBld))) {
        return;
    }

    if (!d.oldObject()) {
        messages.report({2, 3}, "big-bld-created");
    } else if (!d.newObject()) {
        messages.report({2, 3}, "big-bld-deleted");
    } else {
        if (d.geomChanged()) {
            messages.report({2, 3}, "big-bld-geom-changed", Message::Scope::GeomDiff);
        }
        if (d.oldObject()->attr(attr::HEIGHT) != d.newObject()->attr(attr::HEIGHT)) {
            messages.report({2, 3}, "big-bld-height-changed");
        }
    }
}

void checkBldAttrs(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::BLD && diff.oldObject() && diff.newObject())) {
        return;
    }

    const auto oldHeight = diff.oldObject()->attr(attr::HEIGHT).as<int>(-1);
    const auto newHeight = diff.newObject()->attr(attr::HEIGHT).as<int>(-1);

    if (oldHeight > -1 && newHeight > -1 && std::abs(newHeight - oldHeight) >= 50) {
        messages.report({2, 3}, "bld-big-height-change");
    }

    const auto oldCondition = getCondition(*diff.oldObject());
    const auto newCondition = getCondition(*diff.newObject());

    if (Condition::UnderConstruction == oldCondition && Condition::Normal == newCondition) {
        messages.report({3, 3}, "bld-condition-under-construction-to-normal-switched");
    }

    if (Condition::Demolished != oldCondition && Condition::Demolished == newCondition) {
        messages.report({3, 2}, "bld-condition-set-to-demolished");
    }
}

void checkBldWithImportantPoi(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::BLD &&
            (diff.attrsChanged() || diff.geomChanged()))) {
        return;
    }

    auto isImportantBld = [](const Object& obj, Snapshot& snapshot)
    {
        const auto* envelopePtr = obj.geom()->getEnvelopeInternal();
        ASSERT(envelopePtr);
        auto envelope = *envelopePtr;

        auto intersecting =
            snapshot.primitivesByEnvelope(envelope, GeometryType::Point, {});

        for (const auto& objPtr : intersecting) {
            if (isNmapsPoi(*objPtr) &&
                isMajorPoi(*objPtr) &&
                objPtr->geom()->intersects(obj.geom().geosGeometryPtr())) {
                    return true;
            }
            if ((objPtr->categoryId() == cat::TRANSPORT_TERMINAL ||
                        common::isIn(getFtType(*objPtr), {
                            FtType::TransportRailwayTerminal,
                            FtType::TransportAirport,
                            FtType::TransportWaterwayRiverport})) &&
                    objPtr->geom()->intersects(obj.geom().geosGeometryPtr())) {
                return true;
            }
        }
        return false;
    };

    if (diff.newObject() &&
            isImportantBld(*diff.newObject(), diff.newSnapshot())) {
        if (diff.oldObject()) {
            messages.report({1, 1}, "bld-with-important-poi-changed");
        } else {
            messages.report({1, 1}, "bld-with-important-poi-created");
        }
    } else if (diff.oldObject() &&
            isImportantBld(*diff.oldObject(), diff.oldSnapshot())) {
        if (diff.newObject()) {
            messages.report({1, 1}, "bld-with-important-poi-changed");
        } else {
            messages.report({1, 1}, "bld-with-important-poi-deleted");
        }
    }
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
