#include "rd.h"
#include "magic_string.h"
#include "utils.h"
#include "../message_reporter.h"

#include <yandex/maps/wiki/diffalert/object.h>
#include <yandex/maps/wiki/common/rd/cond_type.h>

#include <string>

namespace maps {
namespace wiki {
namespace diffalert {
namespace {

common::ConditionType getCondType(const Object& object) {
    return static_cast<common::ConditionType>(object.attr(attr::COND_TYPE).as<int>());
}

} // namespace

void checkForbiddenCond(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::COND && diff.newObject())) {
        return;
    }

    auto isSuspicious = [](AccessId id) {
        return id == AccessId::None || id == AccessId::Pedestrian;
    };

    if (common::ConditionType::Prohibited == getCondType(*diff.newObject())) {
        if (isSuspicious(getAccessId(*diff.newObject()))) {
            messages.report({3, 4}, "cond-forbidden-suspicious-access-id");
        }
    }
}

void checkCondTollModified(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::COND_TOLL)) {
        return;
    }

    if (!diff.newObject()) {
        messages.report({0, 0}, "cond-toll-deleted");
        return;
    }

    if (!diff.oldObject()) {
        messages.report({0, 0}, "cond-toll-created");
        return;
    }

    if (diff.attrsChanged() || diff.tableAttrsChanged()) {
        messages.report({1, 0}, "cond-toll-attributes-modified");
    }

    if (diff.relationsChanged()) {
        messages.report({1, 0}, "cond-toll-relations-modified");
    }
}

void checkCondAnnotationModified(const DiffContext& diff, MessageReporter& messages)
{
    if (diff.categoryId() != cat::COND_ANNOTATION) {
        return;
    }

    if (!diff.newObject()) {
        messages.report({0, 0}, "cond-annotation-deleted");
        return;
    }

    if (!diff.oldObject()) {
        messages.report({0, 0}, "cond-annotation-created");
        return;
    }

    if (diff.attrsChanged()) {
        messages.report({0, 0}, "cond-annotation-attributes-modified");
    }

    if (diff.relationsChanged()) {
        messages.report({0, 0}, "cond-annotation-relations-modified");
    }
}

void checkCondRdElMoved(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::RD_EL && diff.geomChanged())
        || !diff.newObject()
        || !diff.oldObject()) {
        return;
    }
    const auto masters = diff.newObject()->loadMasterRelations();
    TIds condIds;
    for (const auto& relation : masters) {
        if (relation.role == role::FROM || relation.role == role::TO) {
            condIds.insert(relation.masterId);
        }
    }
    if (condIds.empty()) {
        return;
    }
    const auto conds = diff.newSnapshot().objectsByIds(condIds);
    for (const auto& cond : conds) {
        if (cond->categoryId() == cat::COND_ANNOTATION) {
            messages.report({0, 0}, "rd-el-with-cond-annotation-moved");
            return;
        }
    }
}

void checkCondRdJcMoved(const DiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::RD_JC && diff.geomChanged())
        || !diff.newObject()
        || !diff.oldObject()) {
        return;
    }
    const auto masters = diff.newObject()->loadMasterRelations();
    TIds condIds;
    for (const auto& relation : masters) {
        if (relation.role == role::VIA) {
            condIds.insert(relation.masterId);
        }
    }
    if (condIds.empty()) {
        return;
    }
    const auto conds = diff.newSnapshot().objectsByIds(condIds);
    for (const auto& cond : conds) {
        if (cond->categoryId() == cat::COND_TOLL) {
            messages.report({1, 0}, "rd-jc-with-cond-toll-moved");
            return;
        }
        if (cond->categoryId() == cat::COND_ANNOTATION) {
            messages.report({0, 0}, "rd-jc-with-cond-annotation-moved");
            return;
        }
    }
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
