#include "flat_range.h"
#include "utils.h"
#include "magic_string.h"

#include <yandex/maps/wiki/diffalert/object.h>
#include <yandex/maps/wiki/diffalert/snapshot.h>

#include <maps/wikimap/mapspro/libs/flat_range/include/range.h>
#include <maps/wikimap/mapspro/libs/flat_range/include/validation.h>

#include <string>
#include <unordered_set>
#include <vector>

namespace maps::wiki::diffalert {

namespace {

const std::map<flat_range::ValidationResult, std::string> VALIDATION_RESULT_TO_ALERT =
{
    {flat_range::ValidationResult::BadFlatRange, "entrance-bad-flat-range-assigned"},
    {flat_range::ValidationResult::BadLevelRange, "entrance-bad-level-range-assigned"},
    {flat_range::ValidationResult::FlatRangesIntersection, "entrance-flat-ranges-intersect"},
    {flat_range::ValidationResult::FlatsOutOfRange, "entrance-flats-out-of-range"},
    {flat_range::ValidationResult::LevelsOutOfRange, "entrance-levels-out-of-range"},
    {flat_range::ValidationResult::TooManyFlatsPerEntrance, "too-many-flats-per-entrance"},
    {flat_range::ValidationResult::FlatsReverseOrder, "entrance-flats-reverse-order"},
    {flat_range::ValidationResult::LevelsReverseOrder, "entrance-levels-reverse-order"},
};

std::unordered_set<std::string>
getMessages(
    const std::vector<Snapshot::ObjectPtr>& flatRangeObjects)
{
    std::unordered_set<std::string> result;

    std::vector<flat_range::FlatLevelRange> flatLevelRanges;
    for (const auto& object : flatRangeObjects) {
        flatLevelRanges.push_back({
            object->attr(attr::FLAT_RANGE_FLATS).value(),
            object->attr(attr::FLAT_RANGE_LEVELS).value()
        });
    }

    auto validationResult = flat_range::validate(flatLevelRanges);
    for (const auto& problem : validationResult) {
        ASSERT(VALIDATION_RESULT_TO_ALERT.contains(problem));
        result.insert(VALIDATION_RESULT_TO_ALERT.at(problem));
    }

    return result;
}

} // namespace

void checkEntranceFlatRanges(const DiffContext& diff, MessageReporter& messages)
{
    if (diff.categoryId() != cat::POI_ENTRANCE ||
        !diff.newObject())
    {
        return;
    }

    const auto oldFlatRangeIds = diff.oldObject()
        ? slaveRelativesIds(*diff.oldObject(), {role::FLAT_RANGE_ASSIGNED})
        : TIds();
    const auto newFlatRangeIds = slaveRelativesIds(
        *diff.newObject(), {role::FLAT_RANGE_ASSIGNED});

    const auto oldObjects = diff.oldSnapshot().objectsByIds(oldFlatRangeIds);
    const auto newObjects = diff.newSnapshot().objectsByIds(newFlatRangeIds);

    const auto oldMessages = getMessages(oldObjects);
    const auto newMessages = getMessages(newObjects);

    for (const auto& newMessage : newMessages) {
        if (!oldMessages.contains(newMessage)) {
            messages.report({3, 0}, newMessage);
        }
    }
}

} // namespace maps::wiki::diffalert
