#include "poi.h"
#include "magic_string.h"
#include "utils.h"

#include "../message_reporter.h"

#include <maps/wikimap/mapspro/libs/poi_feed/include/helpers.h>
#include <yandex/maps/wiki/diffalert/snapshot.h>
#include <yandex/maps/wiki/diffalert/diff_context.h>
#include <yandex/maps/wiki/common/misc.h>

#include <algorithm>
#include <cstdint>

namespace maps::wiki::diffalert {

void checkMajorPoi(const DiffContext& diff, MessageReporter& messages, RunMode runMode)
{
    if (!(isPoi(diff.categoryId()) && isAny(diff, isMajorPoi))) {
        return;
    }
    const auto respGroup = teamResponsibilityGroup(diff);
    if (!diff.oldObject()) {
        messages.report(prioByRespGroup({1, 1}, respGroup), makePoiMessage("major-poi-created", diff, runMode));
    } else if (!diff.newObject()) {
        messages.report(prioByRespGroup({1, 1}, respGroup), makePoiMessage("major-poi-deleted", diff, runMode));
    } else {
        if (diff.categoryChanged() && respGroup == TeamResponsibilityGroup::Nmaps) {
            messages.report(prioByRespGroup({1, 1}, respGroup), makePoiMessage("major-poi-category-changed", diff, runMode));
        }
        if (diff.attrsChanged() || diff.tableAttrsChanged()) {
            messages.report(prioByRespGroup({1, 1}, respGroup), makePoiMessage("major-poi-attrs-changed", diff, runMode));
        }
        if (diff.geomChanged()) {
            const auto& oldGeom = diff.oldObject()->geom();
            const auto& newGeom = diff.newObject()->geom();
            const double distanceMeter = mercatorDistanceRatio(oldGeom) * oldGeom.distance(newGeom);
            if (distanceMeter > 10) {
                messages.report(
                    prioByRespGroup({1, 2}, respGroup),
                    makePoiMessage("major-poi-displaced", diff, runMode),
                    Message::Scope::GeomDiff);
            }
        }
    }
}

void checkPoiAssign(const DiffContext& diff, MessageReporter& messages, RunMode runMode)
{
    if (!(isPoi(diff.categoryId()) && diff.newObject())) {
        return;
    }
    const auto respGroup = teamResponsibilityGroup(diff);
    if (respGroup != TeamResponsibilityGroup::Nmaps) {
        return;
    }
    auto getUrbanIds = [](const Relations& relations)
    {
        TIds ids;
         for (const auto& relation: relations) {
            if (relation.role == role::URBAN_AREAL_ASSIGNED) {
                ids.insert(relation.slaveId);
            }
        }
        return ids;
    };

    auto isDistantFrom = [](const Geom& lhs, const Geom& rhs)
    {
        const double ratio = mercatorDistanceRatio(lhs);
        return ratio * lhs.distance(rhs) > 2000;
    };

    if (diff.relationsChanged()) {
        const TIds urbanIds = getUrbanIds(diff.relationsAdded());
        for (const auto& urban: diff.newSnapshot().objectsByIds(urbanIds)) {
            if (isDistantFrom(diff.newObject()->geom(), urban->geom())) {
                messages.report(prioByRespGroup({3, 2}, respGroup),
                    makePoiMessage("poi-to-distant-area-assigned", diff, runMode));
                return;
            }
        }
    }

    if (diff.geomChanged()) {
        const TIds urbanIds = getUrbanIds(diff.newObject()->loadSlaveRelations());
        for (const auto& urban: diff.newSnapshot().objectsByIds(urbanIds)) {
            if (isDistantFrom(diff.newObject()->geom(), urban->geom())) {
                messages.report(prioByRespGroup({3, 2}, respGroup),
                    makePoiMessage("poi-distant-from-area-moved", diff, runMode),
                          Message::Scope::GeomDiff);
                return;
            }
        }
    }
}

void checkPoiNames(const DiffContext& diff, MessageReporter& messages, RunMode runMode)
{
    if (!(isPoi(diff.categoryId()) && diff.newObject())) {
        return;
    }
    const auto respGroup = teamResponsibilityGroup(diff);
    if (respGroup != TeamResponsibilityGroup::Nmaps) {
        return;
    }

    const bool anyOfficialNameAdded = std::any_of(
        diff.tableAttrsAdded().begin(), diff.tableAttrsAdded().end(),
        [&](const Relation& relation) {
            return relation.role == role::name::OFFICIAL;
        }
    );

    const bool anyRenderLabelDeleted = std::any_of(
        diff.tableAttrsDeleted().begin(), diff.tableAttrsDeleted().end(),
        [&](const Relation& relation) {
            return relation.role == role::name::RENDER_LABEL;
        }
    );

    if (!anyOfficialNameAdded && !anyRenderLabelDeleted) {
        return;
    }

    TIds officialNameIds;
    TIds renderLabelIds;
    for (const auto& relation: diff.newObject()->loadSlaveRelations()) {
        if (relation.role == role::name::OFFICIAL) {
            officialNameIds.insert(relation.slaveId);
        } else if (relation.role == role::name::RENDER_LABEL) {
            renderLabelIds.insert(relation.slaveId);
        }
    }

    const auto names = diff.newSnapshot().objectsByIds(join(officialNameIds, renderLabelIds));

    std::set<std::string> langWithRenderLabel;
    for (const auto& name: names) {
        if (renderLabelIds.count(name->id())) {
            langWithRenderLabel.insert(name->attr(attr::LANG).value());
        }
    }

    for (const auto& name: names) {
        if (officialNameIds.count(name->id()) && utf8length(name->attr(attr::NAME)) >= 55) {
            if (!langWithRenderLabel.count(name->attr(attr::LANG).value())) {
                messages.report(prioByRespGroup({3, 4}, respGroup),
                    makePoiMessage("poi-has-long-official-name-and-no-render-label", diff, runMode));
            }
        }
    }
}

void checkMajorPoiLongTask(const DiffContext& diff, MessageReporter& messages)
{
    checkMajorPoi(diff, messages, RunMode::LongTask);
}

void checkPoiAssignLongTask(const DiffContext& diff, MessageReporter& messages)
{
    checkPoiAssign(diff, messages, RunMode::LongTask);
}

void checkPoiNamesLongTask(const DiffContext& diff, MessageReporter& messages)
{
    checkPoiNames(diff, messages, RunMode::LongTask);
}

void checkMajorPoiModeration(const DiffContext& diff, MessageReporter& messages)
{
    checkMajorPoi(diff, messages, RunMode::Moderation);
}

void checkPoiAssignModeration(const DiffContext& diff, MessageReporter& messages)
{
    checkPoiAssign(diff, messages, RunMode::Moderation);
}

void checkPoiNamesModeration(const DiffContext& diff, MessageReporter& messages)
{
    checkPoiNames(diff, messages, RunMode::Moderation);
}

namespace {



std::string poiPositionQuality(const OptionalObject& obj)
{
    if (!obj) {
        return {};
    }

    auto positionQuality = obj->attr(attr::POI_POSITION_QUALITY).value();
    return positionQuality.empty()
        ? obj->attr(attr::INDOOR_POI_POSITION_QUALITY).value()
        : positionQuality;
}

bool hasVerifiedCoords(const DiffContext& diff)
{
    auto positionQuality = diff.oldObject()
        ? poiPositionQuality(diff.oldObject())
        : poiPositionQuality(diff.newObject());
    return poi_feed::isVerifiedPositionQuality(positionQuality);
}

bool anyOfficialNameChanged(const DiffContext& diff)
{
    auto anyOfficialNameAdded = std::any_of(
        diff.tableAttrsAdded().begin(), diff.tableAttrsAdded().end(),
        [&](const Relation& relation) {
            return relation.role == role::name::OFFICIAL;
        });

    if (anyOfficialNameAdded) {
        return true;
    }

    auto anyOfficialNameDeleted = std::any_of(
        diff.tableAttrsDeleted().begin(), diff.tableAttrsDeleted().end(),
        [&](const Relation& relation) {
            return relation.role == role::name::OFFICIAL;
        });

    return anyOfficialNameDeleted;
}

} // namespace

void checkPoiWithPositionQuality(const DiffContext& diff, MessageReporter& messages)
{
    if (!isPoi(diff.categoryId()) || !diff.oldObject() || !diff.newObject()) {
        return;
    }
    if (!hasVerifiedCoords(diff)) {
        return;
    }

    const auto importance = std::max(
        poiImportance(*diff.oldObject()),
        poiImportance(*diff.newObject()));

    if (diff.geomChanged()) {
        messages.report(
            prioByRespGroup(poiMessagePriority(importance), diff),
            message::POI_WITH_PRECISE_LOCATION_DISPLACED);
    }

    if (anyOfficialNameChanged(diff)) {
        messages.report(
            prioByRespGroup(poiMessagePriority(importance), diff),
            message::POI_WITH_PRECISE_LOCATION_OFFICIAL_NAME_CHANGED);
    }

    auto oldBusinessId = diff.oldObject()->attr(attr::POI_BUSINESS_ID).value();
    auto newBusinessId = diff.newObject()->attr(attr::POI_BUSINESS_ID).value();

    if (oldBusinessId != newBusinessId) {
        messages.report(
            prioByRespGroup(poiMessagePriority(importance), diff),
            message::POI_WITH_PRECISE_LOCATION_BUSINESS_ID_CHANGED);
    }

    auto oldBusinessRubricId = diff.oldObject()->attr(attr::POI_BUSINESS_RUBRIC_ID).value();
    auto newBusinessRubricId = diff.newObject()->attr(attr::POI_BUSINESS_RUBRIC_ID).value();

    if (oldBusinessRubricId != newBusinessRubricId) {
        messages.report(
            prioByRespGroup(poiMessagePriority(importance), diff),
            message::POI_WITH_PRECISE_LOCATION_RUBRIC_ID_CHANGED);
    }
}

} // namespace maps::wiki::diffalert
