#include "ad.h"
#include "../checks/utils.h"
#include "../checks/magic_string.h"

#include "../message_reporter.h"

#include <yandex/maps/wiki/diffalert/revision/snapshot.h>
#include <yandex/maps/wiki/diffalert/revision/diff_context.h>

#include <algorithm>
#include <string>

namespace maps::wiki::diffalert {

namespace {

bool isHighPriorityAd(const Object& object)
{
    const auto levelKind = getLevelKind(object);
    const auto informal = object.attr(attr::INFORMAL).as<bool>();
    return levelKind == LevelKind::Country
        || levelKind == LevelKind::Province
        || levelKind == LevelKind::Area
        || (levelKind == LevelKind::Locality && !informal)
        || levelKind == LevelKind::District;
}

bool isInformalLocality(const Object& object)
{
    const auto levelKind = getLevelKind(object);
    const auto informal = object.attr(attr::INFORMAL).as<bool>();
    return levelKind == LevelKind::Locality && informal;
}

TId getSubstitutedAdId(const Relations& relations, TId adSubstId)
{
    for (const auto& relation : relations) {
        if (adSubstId == relation.slaveId && relation.role == role::SUBSTITUTION) {
            return relation.masterId;
        }
    }
    return 0;
}

} // namespace

void checkAdSignificantAreaChange(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::AD && diff.oldObject() && diff.newObject() &&
            diff.geomChanged())) {
        return;
    }

    auto oldArea = contourObjectArea(diff, SnapshotTime::Old);
    auto newArea = contourObjectArea(diff, SnapshotTime::New);

    auto check = [](double oldValue, double newValue)
    {
        return newValue > oldValue * 2.0 || newValue < oldValue * 0.5;
    };

    if (check(oldArea.exterior, newArea.exterior) ||
        check(oldArea.interior, newArea.interior) ||
        check(oldArea.exterior - oldArea.interior, newArea.exterior - newArea.interior)) {

        uint32_t majorPriority = 3; //Other or Block
        if (isAny(diff, isHighPriorityAd)) {
            majorPriority = 1;
        } else if (isAny(diff, isInformalLocality)) {
            majorPriority = 2;
        }

        double sortPriority = -symDiffArea(diff);
        messages.report({majorPriority, 2, sortPriority}, "ad-significant-area-change",
                        Message::Scope::GeomDiff);
    }
}

void checkAdSubst(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (diff.categoryId() != cat::AD_SUBST) {
        return;
    }
    if (diff.newObject() && !diff.oldObject()) {
        messages.report({0, 0}, "ad-subst-created");
        return;
    }
    if (!diff.newObject() && diff.oldObject()) {
        messages.report({0, 0}, "ad-subst-deleted");
        return;
    }

    const auto oldParentId = getSubstitutedAdId(diff.relationsDeleted(), diff.objectId());
    const auto newParentId = getSubstitutedAdId(diff.relationsAdded(), diff.objectId());
    if (oldParentId != newParentId) {
        messages.report({0, 0}, "ad-substitution-changed");
    }

    const auto& newObject = diff.newObject();
    const auto& oldObject = diff.oldObject();

    if ((oldObject->attr(attr::RECOGNITION)
          == newObject->attr(attr::RECOGNITION)) &&
        (diff.attrsChanged() || diff.tableAttrsChanged()))
    {
        messages.report({1, 0}, "ad-subst-attributes-changed");
    }
    if (diff.geomChanged()) {
        messages.report({1, 0}, "ad-subst-geometry-changed");
    }
}

} // namespace maps::wiki::diffalert
