#include "indoor.h"
#include "../message_reporter.h"
#include "../../checks/magic_string.h"
#include "../../checks/utils.h"

#include <yandex/maps/wiki/diffalert/diff_context.h>
#include <yandex/maps/wiki/diffalert/snapshot.h>

#include <unordered_set>

namespace maps::wiki::diffalert {

namespace {

const std::unordered_set<FtType> LOW_PRIORITY_TYPES =
{
    FtType::IndoorAreaPublic,
    FtType::IndoorAreaBusiness,
    FtType::IndoorAreaService,
    FtType::IndoorAreaVoid,
    FtType::IndoorAreaParking,
    FtType::IndoorAreaRestricted,
    FtType::IndoorInfraElevator,
    FtType::IndoorInfraEscalator,
    FtType::IndoorInfraTravolator,
    FtType::IndoorInfraStairs,
    FtType::IndoorInfraEmergencyExit,
    FtType::IndoorInfraFireStairs,
    FtType::IndoorInfraControlFrame,
    FtType::IndoorInfraLuggageInspection,
    FtType::IndoorInfraPandus,
    FtType::IndoorInfoArrival,
    FtType::IndoorInfoDeparture,
    FtType::IndoorInfoStand,
    FtType::IndoorInfoScheme,
    FtType::IndoorInfoInquiryOffice,
    FtType::IndoorServiceWardrobe,
    FtType::IndoorServiceFittingRoom,
    FtType::IndoorServiceLuggageStorage,
    FtType::IndoorServiceTicketOffice,
    FtType::IndoorServiceSmokingRoom,
    FtType::IndoorServiceWheelchairRent,
    FtType::IndoorServiceChargingSocket,
    FtType::IndoorServiceInstantPhoto,
    FtType::IndoorServiceWaitingRoom,
    FtType::IndoorServiceShoppingCartParking,
    FtType::IndoorServiceVendingMachine,
    FtType::IndoorLeisureCinemaHall,
    FtType::IndoorLeisureSlotMachine,
    FtType::IndoorLeisureScene,
    FtType::IndoorLeisureAquarium,
    FtType::IndoorLeisureMassageChair,
};

bool isBoundToOperatingPlan(const DiffContext& diff)
{
    auto poi = diff.oldObject() ? diff.oldObject() : diff.newObject();
    auto& snapshot = diff.oldObject() ? diff.oldSnapshot() : diff.newSnapshot();
    ASSERT(poi);
    const auto levelRelations = poi->loadMasterRelations();
    ASSERT(levelRelations.size() == 1);
    auto levels = snapshot.objectsByIds({levelRelations.begin()->masterId});
    ASSERT(!levels.empty());
    const auto planRelations = levels[0]->loadMasterRelations();
    ASSERT(planRelations.size() == 1);
    auto plans = snapshot.objectsByIds({planRelations.begin()->masterId});
    ASSERT(!plans.empty());
    return plans[0]->attr(attr::SYS_NOT_OPERATING).value().empty();
}

bool isPoint(const DiffContext& diff)
{
    return
        (diff.oldObject() && !diff.oldObject()->geom().isNull() &&
            diff.oldObject()->geom()->getGeometryTypeId() == geos::geom::GEOS_POINT) ||
        (diff.newObject() && !diff.newObject()->geom().isNull() &&
            diff.newObject()->geom()->getGeometryTypeId() == geos::geom::GEOS_POINT);
}

bool isPolygon(const DiffContext& diff)
{
    return
        (diff.oldObject() && !diff.oldObject()->geom().isNull() &&
            diff.oldObject()->geom()->getGeometryTypeId() == geos::geom::GEOS_POLYGON) ||
        (diff.newObject() && !diff.newObject()->geom().isNull() &&
            diff.newObject()->geom()->getGeometryTypeId() == geos::geom::GEOS_POLYGON);
}

bool isLowPriority(const DiffContext& diff)
{
    return
        !isPoint(diff) ||
        (
            (!diff.oldObject() || LOW_PRIORITY_TYPES.count(getFtType(*diff.oldObject()))) &&
            (!diff.newObject() || LOW_PRIORITY_TYPES.count(getFtType(*diff.newObject())))
        );
}

Priority selectPriority(const DiffContext& diff, uint32_t majorHigh, uint32_t majorLow)
{
    return isLowPriority(diff) ? Priority {majorLow, 0} : Priority {majorHigh, 0};
}

bool isGeomModified(const DiffContext& diff)
{
    if (!diff.newObject() || !diff.oldObject()) {
        return false;
    }
    const auto& oldGeom = diff.oldObject()->geom();
    const auto& newGeom = diff.newObject()->geom();
    if (oldGeom.isNull() || newGeom.isNull()) {
        return false;
    }
    if (isPolygon(diff)) {
        if (oldGeom->disjoint(newGeom.geosGeometryPtr())) {
            return true;
        }
        Geom intersection(oldGeom->intersection(newGeom.geosGeometryPtr()));
        return intersection->getArea() < 0.5 * oldGeom->getArea();
    }
    return mercatorDistanceRatio(oldGeom) * oldGeom.distance(newGeom) >= 10.0;
}

} // namespace


void checkIndoorCreatedOrDeleted(const DiffContext& diff, MessageReporter& messages)
{
    if (!isIndoor(diff.categoryId()) || (diff.newObject() && diff.oldObject())) {
        return;
    }
    if (!isBoundToOperatingPlan(diff)) {
        return;
    }
    const auto priority = selectPriority(diff, 0, 2);
    if (!diff.oldObject()) {
         messages.report(priority, "indoor-object-created");
    } else if (!diff.newObject()) {
        messages.report(priority, "indoor-object-deleted");
    }
}

void checkIndoorAttrsModified(const DiffContext& diff, MessageReporter& messages)
{
    if (!isIndoor(diff.categoryId()) || !diff.newObject() || !diff.oldObject()) {
        return;
    }
    if (!isBoundToOperatingPlan(diff)) {
        return;
    }
    const auto priority = selectPriority(diff, 1, 2);
    if (diff.attrsChanged() || diff.tableAttrsChanged()) {
        messages.report(priority, "indoor-object-attributes-modified");
    }
}

void checkIndoorGeometryModified(const DiffContext& diff, MessageReporter& messages)
{
    if (!isIndoor(diff.categoryId()) || !diff.newObject() ||!diff.oldObject()) {
        return;
    }
    if (!isBoundToOperatingPlan(diff)) {
        return;
    }
    if (!isGeomModified(diff)) {
        return;
    }
    const auto priority = selectPriority(diff, 1, 2);
    messages.report(priority, "indoor-object-geometry-modified");
}

} // namespace maps::wiki::diffalert
