#include "parkings.h"

#include "../checks/magic_string.h"
#include "../checks/utils.h"
#include "../message_reporter.h"

#include <geos/algorithm/distance/DiscreteHausdorffDistance.h>

#include <yandex/maps/wiki/diffalert/revision/diff_context.h>

#include <algorithm>

namespace maps {
namespace wiki {
namespace diffalert {
namespace {

const auto RESTRICTED_PARKING_MAX_LENGTH_IN_METERS = 20.0;

TId getParentZoneId(TId linearParkingLotId, const Relations& relations)
{
    for (const auto& relation: relations) {
        if (linearParkingLotId == relation.slaveId &&
            relation.role == role::PARKING_LOT_LINEAR_ASSIGNED)
        {
            return relation.masterId;
        }
    }

    return 0;
};

bool isParkingLotLinear(const LongtaskDiffContext& diff)
{
    return diff.categoryId() == cat::URBAN_ROADNET_PARKING_LOT_LINEAR;
}

bool isParkingLot(const LongtaskDiffContext& diff)
{
    return diff.categoryId() == cat::URBAN_ROADNET_PARKING_LOT;
}

bool isParkingLotLinearFree(const Object& object)
{
    return getFtType(object) == FtType::UrbanRoadnetParkingFree;
}

bool isParkingLotLinearToll(const Object& object)
{
    return getFtType(object) == FtType::UrbanRoadnetParkingToll;
}

bool isParkingLotLinearRestricted(const Object& object)
{
    return getFtType(object) == FtType::UrbanRoadnetParkingRestricted;
}

bool isParkingLotConnectedWithUrbanAreal(const Relations& relations)
{
    return std::any_of(
        relations.cbegin(), relations.cend(),
        [](const Relation& relation) {
            return relation.role == role::URBAN_AREAL_ASSIGNED;
        }
    );
};

bool isParkingLotInBld(const Object& object) {
    return static_cast<bool>(
        object.attr(attr::URBAN_ROADNET_PARKING_LOT_BLD)
    );
}

} // namespace

void checkParkingLotLinearAttrsChanged(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (isParkingLotLinear(diff) &&
        diff.oldObject() && diff.newObject() &&
        isAny(diff, isParkingLotLinearToll) &&
        diff.attrsChanged())
    {
        messages.report({2, 0}, "parking-lot-linear-toll-attributes-changed");
    }
}

void checkParkingLotLinearCreated(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (diff.oldObject() || !diff.newObject() || !isParkingLotLinear(diff)) {
        return;
    }

    const auto parking = diff.newObject();

    if (isParkingLotLinearFree(*parking)) {
        messages.report({2, 0}, "parking-lot-linear-free-created");
        return;
    }

    if (isParkingLotLinearToll(*parking)) {
        messages.report({2, 0}, "parking-lot-linear-toll-created");
        return;
    }

    if (isParkingLotLinearRestricted(*parking)) {
        if (parking->geom().realLength() > RESTRICTED_PARKING_MAX_LENGTH_IN_METERS) {
            messages.report({2, 0}, "parking-lot-linear-restricted-long-created");
        }
        return;
    }
}

void checkParkingLotLinearDeleted(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (isParkingLotLinear(diff) &&
        diff.oldObject() && !diff.newObject())
    {
        if (isParkingLotLinearToll(*diff.oldObject())) {
            messages.report({2, 0}, "parking-lot-linear-toll-deleted");
        }
    }
}

void checkParkingLotLinearZoneChanged(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (isParkingLotLinear(diff) &&
        diff.oldObject() && diff.newObject() &&
        diff.relationsChanged())
    {
        if (!isParkingLotLinearToll(*diff.newObject())) {
            return;
        }

        const auto oldParentId = getParentZoneId(diff.objectId(), diff.relationsDeleted());
        const auto newParentId = getParentZoneId(diff.objectId(), diff.relationsAdded());

        if (oldParentId != newParentId) {
            messages.report({2, 0}, "parking-lot-linear-toll-zone-changed");
        }
    }
}

void checkParkingLotLinearGeometrySignificantlyChanged(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (isParkingLotLinear(diff) &&
        diff.oldObject() && diff.newObject() &&
        diff.geomChanged())
    {
        const auto MAX_DISTANCE_IN_METERS = 50.0;

        const auto parking = diff.newObject();
        const auto mercatorRatio = mercatorDistanceRatio(parking->geom());
        const auto distance =
            geos::algorithm::distance::DiscreteHausdorffDistance::distance(
                *diff.oldObject()->geom().geosGeometryPtr(),
                *diff.newObject()->geom().geosGeometryPtr()
            );

        if (distance * mercatorRatio > MAX_DISTANCE_IN_METERS) {
            if (isParkingLotLinearFree(*parking)) {
                messages.report({2, 0}, "parking-lot-linear-free-geometry-significantly-changed");
            } else if (isParkingLotLinearToll(*parking)) {
                messages.report({2, 0}, "parking-lot-linear-toll-geometry-significantly-changed");
            } else if (isParkingLotLinearRestricted(*parking)) {
                messages.report({2, 0}, "parking-lot-linear-restricted-geometry-significantly-changed");
            }
        }
    }
}

void checkParkingLotCreated(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (isParkingLot(diff) &&
        !diff.oldObject() && diff.newObject())
    {
        auto parking = diff.newObject();

        if (isParkingLotConnectedWithUrbanAreal(diff.relationsAdded())) {
            messages.report({2, 0}, "parking-lot-connected-with-territory-created");
        }

        if (isParkingLotInBld(*parking)) {
            messages.report({2, 0}, "parking-lot-in-bld-created");
        }
    }
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
