#include "rd_el.h"
#include "../checks/magic_string.h"
#include "../checks/utils.h"
#include "../message_reporter.h"

#include <yandex/maps/wiki/diffalert/revision/snapshot.h>
#include <yandex/maps/wiki/diffalert/revision/diff_context.h>


#if YANDEX_MAPS_UBUNTU_VERSION == 1004
// fix boost 1.40
#include "date_formatting_lucid.hpp"
#endif

#include <boost/date_time/gregorian/gregorian.hpp>
#include <stdexcept>

namespace maps {
namespace wiki {
namespace diffalert {

namespace {

boost::gregorian::date extractOpenDate(const Object& object)
{
    auto openAttr = object.attr(attr::SRV_UC_OPEN_AT);
    if (!openAttr.as<bool>()) {
        return {};
    }
    try {
        return boost::gregorian::from_undelimited_string(
            openAttr.as<std::string>());
    } catch (const std::exception&) {
        return {};
    }
}

const boost::gregorian::days OPEN_SOON_DAYS_HIGH{30};
const Priority OPEN_SOON_PRIORITY_HIGH{0, 2};

const boost::gregorian::days OPEN_SOON_DAYS_LOW{60};
const Priority OPEN_SOON_PRIORITY_LOW{3, 2};

} // namespace

void checkRdElUnderConstructionLongTask(
        const LongtaskDiffContext& diff,
        MessageReporter& messages)
{
    if (!(diff.categoryId() == cat::RD_EL && isAny(diff, isUnderConstruction))) {
        return;
    }

    const auto oldUnderConstruction = diff.oldObject() &&
        isUnderConstruction(*diff.oldObject());
    const auto newUnderConstruction = diff.newObject() &&
        isUnderConstruction(*diff.newObject());

    if (!diff.newObject() && oldUnderConstruction) {
        messages.report({1, 0}, "rd-el-srv-uc-deleted");
    } else if (!diff.oldObject() && newUnderConstruction) {
        messages.report({1, 0}, "rd-el-srv-uc-created");
    } else if (oldUnderConstruction != newUnderConstruction) {
        messages.report({1, 0}, "rd-el-srv-uc-changed");
    }

    if (newUnderConstruction) {
        auto openDate = extractOpenDate(*diff.newObject());
        if (openDate.is_not_a_date()) {
            return;
        }

        auto currentDate = boost::gregorian::day_clock::universal_day();
        Priority openSoonPriority{(uint32_t)-1, (uint32_t)-1};
        if (openDate - currentDate < OPEN_SOON_DAYS_HIGH) {
            openSoonPriority = OPEN_SOON_PRIORITY_HIGH;
        } else if (openDate - currentDate < OPEN_SOON_DAYS_LOW) {
            openSoonPriority = OPEN_SOON_PRIORITY_LOW;
        } else {
            return;
        }

        if (diff.attrsChanged()) {
            messages.report(openSoonPriority, "rd-el-open-at-soon-attrs-changed");
        }
        if (diff.geomChanged()) {
            messages.report(openSoonPriority, "rd-el-open-at-soon-geom-changed");
        }
    }
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
