#include "transport.h"
#include "../../checks/utils.h"
#include "../../checks/magic_string.h"

#include "../message_reporter.h"

#include <yandex/maps/wiki/diffalert/revision/snapshot.h>
#include <yandex/maps/wiki/diffalert/revision/diff_context.h>

#include <yandex/maps/wiki/common/misc.h>

namespace maps {
namespace wiki {
namespace diffalert {
namespace {

const std::map<std::string, uint32_t> ANY_CHANGE_MONITORING_TRANSPORT_POINTS_TO_MAJOR {
    {cat::TRANSPORT_STOP, 2},
    {cat::TRANSPORT_HELICOPTER, 1}
};

bool isTransportPoint(const std::string& categoryId)
{
    return common::isIn(categoryId, {
            cat::TRANSPORT_METRO_STATION,
            cat::TRANSPORT_METRO_EXIT,

            cat::TRANSPORT_AIRPORT_TERMINAL,
            cat::TRANSPORT_AIRPORT,
            cat::TRANSPORT_HELICOPTER,

            cat::TRANSPORT_STOP,
            cat::TRANSPORT_TERMINAL,

            cat::TRANSPORT_RAILWAY_STATION,

            cat::TRANSPORT_WATERWAY_STOP,
        }
    );
}

} // namespace


void checkTransportPointDispClassChange(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    if (!(isTransportPoint(diff.categoryId()) && !diff.stateChanged() && diff.attrsChanged())) {
        return;
    }

    if (getDispClass(*diff.oldObject()) != getDispClass(*diff.newObject())) {
        messages.report({2, 2}, "transport-point-disp-class-changed");
    }
}

void checkTransportPointsAnyChange(const LongtaskDiffContext& diff, MessageReporter& messages)
{
    const auto it = ANY_CHANGE_MONITORING_TRANSPORT_POINTS_TO_MAJOR.find(diff.categoryId());
    if (it == ANY_CHANGE_MONITORING_TRANSPORT_POINTS_TO_MAJOR.end()) {
        return;
    }
    const auto major = it->second;

    if (!diff.oldObject()) {
        messages.report({major, 1}, "transport-point-created");
    } else if (!diff.newObject()) {
        messages.report({major, 1}, "transport-point-deleted");
    } else {
        if (diff.geomChanged()) {
            messages.report({major, 1}, "transport-point-geom-changed", Message::Scope::GeomDiff);
        }
        if (namesChanged(diff)) {
            messages.report({major, 1}, "transport-point-names-changed");
        }
    }
}

void checkTramRouteChange(const LongtaskDiffContext& diff, MessageReporter& messages) {
    if (diff.categoryId() != cat::TRANSPORT_TRAM_ROUTE &&
        diff.categoryId() != cat::TRANSPORT_TRAM_EL)
    {
        return;
    }
    if (diff.geomChanged() || diff.relationsChanged()) {
        messages.report({2, 2}, "transport-route-geom-changed", Message::Scope::GeomDiff);
    }
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
