#include "diff_context_impl.h"
#include "snapshot_impl.h"
#include "db_access.h"
#include "geom_index.h"
#include "diff_loaders.h"
#include "extractors.h"
#include "exclusions.h"

#include <yandex/maps/wiki/diffalert/revision/editor_config.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/profiletimer.h>
#include <maps/libs/log8/include/log8.h>
#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/filters.h>
#include <yandex/maps/wiki/revision/snapshot.h>
#include <yandex/maps/wiki/revision/snapshot_id.h>
#include <yandex/maps/wiki/common/retry_duration.h>
#include <yandex/maps/wiki/common/string_utils.h>

#include <future>
#include <map>
#include <set>
#include <string>
#include <utility>

namespace mwr = maps::wiki::revision;
namespace rf = maps::wiki::revision::filters;

namespace maps {
namespace wiki {
namespace diffalert {
namespace {

void checkBranch(const mwr::Branch& branch)
{
    REQUIRE(branch.type() != mwr::BranchType::Approved,
        branch.type() << " branches are not supported");
}

struct DifferencingFilters
{
    rf::ProxyFilterExpr oldSnapshotFilter;
    rf::ProxyFilterExpr newSnapshotFilter;
};

DifferencingFilters makeDifferencingFilters(
        const mwr::Branch& oldBranch,
        const mwr::SnapshotId& oldSnapshotId,
        const mwr::Branch& newBranch,
        const mwr::SnapshotId& newSnapshotId)
{
    if (oldBranch.id() == newBranch.id()) {
        return {
            rf::CommitAttr::id() > newSnapshotId.commitId(),
            rf::CommitAttr::id() > oldSnapshotId.commitId()
        };
    }

    auto makeFilter = [](
            const mwr::Branch& branch,
            const mwr::Branch& otherBranch)
            -> rf::ProxyFilterExpr {
        if (branch.type() == mwr::BranchType::Trunk) {
            return rf::CommitAttr::stableBranchId() > otherBranch.id() ||
                rf::CommitAttr::stableBranchId().isNull();
        } else if (otherBranch.type() == mwr::BranchType::Trunk) {
            return !rf::CommitAttr::isTrunk();
        }
        return rf::CommitAttr::stableBranchId() > otherBranch.id() ||
            !rf::CommitAttr::isTrunk();
    };

    return {
        makeFilter(oldBranch, newBranch),
        makeFilter(newBranch, oldBranch)
    };
}

} // namespace

struct CompareSnapshotsResult::Impl
{
    Impl(
            std::vector<LongtaskDiffContext>&& diffContexts,
            TIds&& badObjects,
            TIds&& badRelations)
        : diffContexts(std::move(diffContexts))
        , badObjects(std::move(badObjects))
        , badRelations(std::move(badRelations))
    {
    }

    std::vector<LongtaskDiffContext> diffContexts;
    TIds badObjects;
    TIds badRelations;
};

CompareSnapshotsResult::CompareSnapshotsResult(Impl&& impl)
    : impl_(new Impl(std::move(impl)))
{ }

CompareSnapshotsResult::CompareSnapshotsResult(CompareSnapshotsResult&&) noexcept = default;
CompareSnapshotsResult& CompareSnapshotsResult::operator=(CompareSnapshotsResult&&) noexcept = default;
CompareSnapshotsResult::~CompareSnapshotsResult() = default;

const std::vector<LongtaskDiffContext>&
CompareSnapshotsResult::diffContexts() const& { return impl_->diffContexts; }

std::vector<LongtaskDiffContext>&&
CompareSnapshotsResult::diffContexts() && { return std::move(impl_->diffContexts); }

const TIds&
CompareSnapshotsResult::badObjects() const { return impl_->badObjects; }

const TIds&
CompareSnapshotsResult::badRelations() const  { return impl_->badRelations; }

//===============================================================

LongtaskDiffContext::LongtaskDiffContext(Impl&& impl)
    : impl_(new Impl(std::move(impl)))
{ }

LongtaskDiffContext::LongtaskDiffContext(LongtaskDiffContext&&) noexcept = default;
LongtaskDiffContext& LongtaskDiffContext::operator=(LongtaskDiffContext&&) noexcept = default;
LongtaskDiffContext::~LongtaskDiffContext() = default;

TId LongtaskDiffContext::objectId() const
{ return impl_->objectDiff.anyObject().id(); }

const std::string& LongtaskDiffContext::categoryId() const
{ return impl_->objectDiff.anyObject().categoryId(); }

OptionalObject LongtaskDiffContext::oldObject() const
{ return OptionalObject(impl_->objectDiff.oldObject.get()); }

OptionalObject LongtaskDiffContext::newObject() const
{ return OptionalObject(impl_->objectDiff.newObject.get()); }

LongtaskSnapshot& LongtaskDiffContext::oldSnapshot() const
{ return impl_->snapshotPair.oldSnapshot; }

LongtaskSnapshot& LongtaskDiffContext::newSnapshot() const
{ return impl_->snapshotPair.newSnapshot; }

bool LongtaskDiffContext::categoryChanged() const
{ return impl_->objectDiff.categoryChanged; }

bool LongtaskDiffContext::attrsChanged() const
{ return impl_->objectDiff.attrsChanged; }

bool LongtaskDiffContext::geomChanged() const
{ return impl_->objectDiff.geomChanged; }

const Relations& LongtaskDiffContext::relationsAdded() const
{ return impl_->relationsDiff.relationsAdded; }

const Relations& LongtaskDiffContext::relationsDeleted() const
{ return impl_->relationsDiff.relationsDeleted; }

const Relations& LongtaskDiffContext::tableAttrsAdded() const
{ return impl_->relationsDiff.tableAttrsAdded; }

const Relations& LongtaskDiffContext::tableAttrsDeleted() const
{ return impl_->relationsDiff.tableAttrsDeleted; }

CompareSnapshotsResult LongtaskDiffContext::compareSnapshots(
        const revision::Branch& oldBranch,
        const revision::SnapshotId& oldSnapshotId,
        const revision::Branch& newBranch,
        const revision::SnapshotId& newSnapshotId,
        pgpool3::Pool& tdsConnPool,
        pgpool3::Pool& viewConnPool,
        const EditorConfig& config,
        const CommitFilter& commitsToExcludeFilter)
{
    checkBranch(oldBranch);
    checkBranch(newBranch);

    auto maxConns = tdsConnPool.state().constants.slaveMaxSize;
    REQUIRE(maxConns >= 4,
        "TDS connection pool is too small; need at least 4 slave connections");

    INFO() << "Starting snapshots comparison: ["
        << oldBranch.id() << ':' << oldSnapshotId << "] => ["
        << newBranch.id() << ':' << newSnapshotId << ']';
    ProfileTimer totalTimer;

    const auto differencingFilters = makeDifferencingFilters(
        oldBranch, oldSnapshotId, newBranch, newSnapshotId);

    RevSnapshotFactory oldRevSnapshotFct(oldBranch, oldSnapshotId, tdsConnPool);
    RevSnapshotFactory newRevSnapshotFct(newBranch, newSnapshotId, tdsConnPool);
    ViewTxnFactory oldViewTxnFct(oldBranch, viewConnPool);
    ViewTxnFactory newViewTxnFct(newBranch, viewConnPool);

    INFO() << "Branch " << oldBranch.id() << " view availability: " << oldViewTxnFct.isAvailable();
    INFO() << "Branch " << newBranch.id() << " view availability: " << newViewTxnFct.isAvailable();

    INFO() << "Build bld with 3d model geom index";

    auto oldBldIndexFuture = std::async(
        std::launch::async,
        makeBldWithModel3dGeomIndex,
        oldRevSnapshotFct
    );

    auto newBldIndexFuture = std::async(
        std::launch::async,
        makeBldWithModel3dGeomIndex,
        newRevSnapshotFct
    );

    const GeomIndexPtr oldBldIndex = oldBldIndexFuture.get();
    const GeomIndexPtr newBldIndex = newBldIndexFuture.get();

    auto makeLongtaskSnapshotPair = [&]() -> LongtaskSnapshotPair {
        return LongtaskSnapshotPair {
            LongtaskSnapshot(LongtaskSnapshot::Impl{
                oldRevSnapshotFct, oldViewTxnFct, oldBldIndex, config}),
            LongtaskSnapshot(LongtaskSnapshot::Impl{
                newRevSnapshotFct, newViewTxnFct, newBldIndex, config}),
        };
    };

    DEBUG() << "Loading differencing revision ids, calculating relation diffs";
    auto relDiffsByOidFuture = std::async(
        std::launch::async, [&]
        {
            auto oldRelRevIdsFuture = std::async(
                std::launch::async, [&]
                {
                    return loadSortedRevisionIds(
                        oldRevSnapshotFct,
                        differencingFilters.oldSnapshotFilter &&
                            rf::ObjRevAttr::isRelation());
                });
            auto newRelRevIds =
                loadSortedRevisionIds(
                    newRevSnapshotFct,
                    differencingFilters.newSnapshotFilter &&
                        rf::ObjRevAttr::isRelation());

            return loadLongtaskRelationDiffs(
                oldRelRevIdsFuture.get(),
                newRelRevIds,
                oldRevSnapshotFct,
                newRevSnapshotFct,
                maxConns - 2);
        });

    auto oldObjRevIdsFuture = std::async(
        std::launch::async, [&]
        {
            return loadSortedRevisionIds(
                oldRevSnapshotFct,
                differencingFilters.oldSnapshotFilter &&
                    rf::ObjRevAttr::isNotRelation());
        });
    auto newObjRevIds = loadSortedRevisionIds(
        newRevSnapshotFct,
        differencingFilters.newSnapshotFilter &&
            rf::ObjRevAttr::isNotRelation());
    auto oldObjRevIds = oldObjRevIdsFuture.get();

    // Wait for pool connections release
    auto relDiffResult = relDiffsByOidFuture.get();

    DEBUG() << "Loading object diffs";

    LongtaskObjectDiffsLoader loader(oldRevSnapshotFct, newRevSnapshotFct, config, maxConns);
    loader.loadChangedObjects(oldObjRevIds, newObjRevIds);

    TIds seenOids;
    for (const auto& revId : oldObjRevIds) {
        seenOids.insert(revId.objectId());
    }
    for (const auto& revId : newObjRevIds) {
        seenOids.insert(revId.objectId());
    }

    {
        std::vector<TId> missingOids;
        for (const auto& oidRelDiffsPair : relDiffResult.relDiffsByOid) {
            auto objectId = oidRelDiffsPair.first;
            if (!seenOids.count(objectId)) {
                missingOids.push_back(objectId);
            }
        }
        loader.loadUnchangedObjects(missingOids);
    }

    DEBUG() << "Building diff contexts, detecting geom part changes";

    std::map<TId, Impl> implsByOid;
    TIds geomChangedGeomPartIds;
    TIds relChangedGeomPartIds;
    TIds badObjects;

    auto loaderResult = loader.getAll();
    badObjects.insert(loaderResult.badObjects.begin(), loaderResult.badObjects.end());
    for (auto&& objDiff : loaderResult.objDiffs) {
        auto objectId = objDiff.anyObject().id();
        auto&& relDiff = relDiffResult.relDiffsByOid[objectId];

        if (isGeomPartCategory(objDiff.anyObject().categoryId())) {
            if (objDiff.geomChanged) {
                geomChangedGeomPartIds.insert(objectId);
            }
            if (!relDiff.relationsAdded.empty() ||
                    !relDiff.relationsDeleted.empty()) {
                relChangedGeomPartIds.insert(objectId);
            }
        }

        implsByOid.insert(
            std::make_pair(
                objectId,
                Impl{makeLongtaskSnapshotPair(), std::move(objDiff), std::move(relDiff)}
            )
        );
    }

    DEBUG() << "Propagating geometry changes to macroobjects";

    while (!geomChangedGeomPartIds.empty() || !relChangedGeomPartIds.empty()) {
        TIds relChangedIdsBuffer;
        TIds geomChangedIdsBuffer;
        std::swap(relChangedGeomPartIds, relChangedIdsBuffer);
        std::swap(geomChangedGeomPartIds, geomChangedIdsBuffer);

        TIds missingMasters;
        auto propagateGeomChanged = [&](const Relation& rel)
        {
            ASSERT(isGeomPartRelation(rel));

            auto masterIt = implsByOid.find(rel.masterId);
            if (masterIt != std::end(implsByOid)) {
                auto& masterObjDiff = masterIt->second.objectDiff;
                auto& masterRelDiff = masterIt->second.relationsDiff;
                if (!masterObjDiff.geomChanged) {
                    masterObjDiff.geomChanged = true;
                    if (isGeomPartCategory(
                            masterObjDiff.anyObject().categoryId())) {
                        geomChangedGeomPartIds.insert(rel.masterId);
                        if (!masterRelDiff.relationsAdded.empty() ||
                                !masterRelDiff.relationsDeleted.empty()) {
                            relChangedGeomPartIds.insert(rel.masterId);
                        }
                    }
                }
            } else {
                missingMasters.insert(rel.masterId);
            }
        };

        // Propagate geom part changes via relation changes
        for (auto objectId : relChangedIdsBuffer) {
            auto contextIt = implsByOid.find(objectId);
            ASSERT(contextIt != std::end(implsByOid));
            const auto& context = contextIt->second;
            for (const auto& rel : context.relationsDiff.relationsAdded) {
                if (rel.slaveId == objectId && isGeomPartRelation(rel)) {
                    propagateGeomChanged(rel);
                }
            }
            for (const auto& rel : context.relationsDiff.relationsDeleted) {
                if (rel.slaveId == objectId && isGeomPartRelation(rel)) {
                    propagateGeomChanged(rel);
                }
            }
        }
        if (!missingMasters.empty()) {
            ERROR() << "Missing masters: " << common::join(missingMasters, ", ");
            badObjects.insert(missingMasters.begin(), missingMasters.end());
        }

        // For geom parts with changed geometry, propagate to geom hierarchy
        if (geomChangedIdsBuffer.empty()) {
            continue;
        }
        {
            auto relRevs = common::retryDuration([&] {
                auto snapshot = newRevSnapshotFct.get();
                return snapshot->loadMasterRelations(geomChangedIdsBuffer);
            });

            for (const auto& relRev : relRevs) {
                auto rel = extractRelation(relRev);
                if (isGeomPartRelation(rel)) {
                    propagateGeomChanged(rel);
                }
            }
        }

        LongtaskObjectDiffsLoader mastersLoader(
            oldRevSnapshotFct, newRevSnapshotFct, config, maxConns);
        mastersLoader.loadUnchangedObjects(
            {missingMasters.begin(), missingMasters.end()});

        auto loaderResult = mastersLoader.getAll();
        badObjects.insert(loaderResult.badObjects.begin(), loaderResult.badObjects.end());
        for (auto&& objDiff : loaderResult.objDiffs) {
            auto objectId = objDiff.anyObject().id();
            objDiff.geomChanged = true;
            if (isGeomPartCategory(objDiff.anyObject().categoryId())) {
                geomChangedGeomPartIds.insert(objectId);
            }
            implsByOid.insert(
                std::make_pair(
                    objectId, Impl{makeLongtaskSnapshotPair(), std::move(objDiff), {}}
                )
            );
        }
    }

    ExclusionFinder exclusionFinder(
        tdsConnPool,
        commitsToExcludeFilter,
        oldBranch,
        oldSnapshotId,
        newBranch,
        newSnapshotId);
    exclusionFinder.loadCommits(
        differencingFilters.oldSnapshotFilter,
        differencingFilters.newSnapshotFilter);
    exclusionFinder.propagateCommitsToMasters(
        implsByOid);

    DEBUG() << "Finalizing diff contexts";

    std::vector<LongtaskDiffContext> contexts;
    contexts.reserve(implsByOid.size());
    for (auto&& oidContextImplPair : implsByOid) {
        LongtaskDiffContext context(std::move(oidContextImplPair.second));
        if (context.changed() && !exclusionFinder.isObjectExcluded(context.objectId())) {
            contexts.push_back(std::move(context));
        }
    }

    INFO() << "Completed snapshots comparison: ["
        << oldBranch.id() << ':' << oldSnapshotId << "] => ["
        << newBranch.id() << ':' << newSnapshotId << "] in "
        << totalTimer.getElapsedTime() << ", found " << contexts.size()
        << " changed objects";

    return CompareSnapshotsResult(
        CompareSnapshotsResult::Impl(
            std::move(contexts),
            std::move(badObjects),
            TIds{relDiffResult.badRelations.begin(), relDiffResult.badRelations.end()}));
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
