#include "diff_context_impl.h"
#include "extractors.h"
#include "object_impl.h"
#include "snapshot_impl.h"
#include "helpers.h"

#include <maps/libs/common/include/exception.h>
#include <yandex/maps/wiki/revision/objectrevision.h>

namespace maps {
namespace wiki {
namespace diffalert {
LongtaskObjectDiff::LongtaskObjectDiff(
        const revision::ObjectRevision* oldRevPtr,
        const RevSnapshotFactory& oldRevSnapshotFct,
        const revision::ObjectRevision* newRevPtr,
        const RevSnapshotFactory& newRevSnapshotFct,
        const EditorConfig& config)
    : categoryChanged(true)
    , attrsChanged(true)
    , geomChanged(false)
{
    ASSERT(oldRevPtr || newRevPtr);

    if (oldRevPtr) {
        ASSERT(!oldRevPtr->data().deleted);
        REQUIRE(oldRevPtr->data().attributes,
            "Object " << oldRevPtr->id() << " has no attributes");
        oldObject.reset(
            new LongtaskObject(LongtaskObject::Impl(*oldRevPtr, oldRevSnapshotFct, config)));
        geomChanged = !!oldRevPtr->data().geometry;
    }
    if (newRevPtr) {
        ASSERT(!newRevPtr->data().deleted);
        REQUIRE(newRevPtr->data().attributes,
            "Object " << newRevPtr->id() << " has no attributes");
        newObject.reset(
            new LongtaskObject(LongtaskObject::Impl(*newRevPtr, newRevSnapshotFct, config)));
        geomChanged = !!newRevPtr->data().geometry;
    }

    if (oldRevPtr && newRevPtr) {
        auto oldCategory = extractCategoryId(*oldRevPtr->data().attributes);
        auto newCategory = extractCategoryId(*newRevPtr->data().attributes);
        categoryChanged = (oldCategory != newCategory);
        attrsChanged = !areAttributesEqual(
            *oldRevPtr->data().attributes, *newRevPtr->data().attributes,
            oldCategory, newCategory);
        geomChanged =
            (oldRevPtr->data().geometry != newRevPtr->data().geometry) ||
                (isInterior(*oldRevPtr) != isInterior(*newRevPtr));
    }
}

LongtaskRelationsDiff& LongtaskRelationsDiff::operator+=(LongtaskRelationsDiff&& rhs)
{
    auto mergeRelations = [](Relations& lhs, Relations&& rhs)
    {
        for (auto&& rel : rhs) {
            lhs.insert(std::move(rel));
        }
    };
    mergeRelations(relationsAdded, std::move(rhs.relationsAdded));
    mergeRelations(relationsDeleted, std::move(rhs.relationsDeleted));
    mergeRelations(tableAttrsAdded, std::move(rhs.tableAttrsAdded));
    mergeRelations(tableAttrsDeleted, std::move(rhs.tableAttrsDeleted));
    return *this;
}

LongtaskDiffContext::Impl::Impl(
        LongtaskSnapshotPair&& snapshotPair_,
        LongtaskObjectDiff&& objectDiff_,
        LongtaskRelationsDiff&& relationsDiff_)
    : snapshotPair(std::move(snapshotPair_))
    , objectDiff(std::move(objectDiff_))
    , relationsDiff(std::move(relationsDiff_))
{ }

} // namespace diffalert
} // namespace wiki
} // namespace maps
