#include <yandex/maps/wiki/diffalert/revision/diff_envelopes.h>
#include "extractors.h"
#include "helpers.h"

#include <maps/libs/log8/include/log8.h>

namespace maps {
namespace wiki {
namespace diffalert {
namespace {

std::map<TId, DiffEnvelopes>
calcDiffEnvelopesById(
        const std::set<TId>& objectIds,
        const std::map<TId, const LongtaskDiffContext*>& diffContextById,
        const revision::Snapshot& oldSnapshot)
{
    std::map<TId, DiffEnvelopes> diffEnvelopesById;

    struct MastersDiff {
        std::set<TId> added;
        std::set<TId> removed;
        std::set<TId> remained;
    };
    std::map<TId, MastersDiff> partIdToMastersDiff;
    std::set<TId> withOwnGeom;

    // First, go down geom-part relations and load diff envelopes of slaves.
    std::set<TId> currentIds = objectIds;
    while (!currentIds.empty()) {
        std::set<TId> nextIds;
        std::set<TId> toLoadRev;
        std::set<TId> toLoadPartIds;

        for (TId oid : currentIds) {
            auto it = diffContextById.find(oid);
            if (it == diffContextById.end()) {
                toLoadRev.insert(oid);
                continue;
            }
            const DiffContext& diffContext = *it->second;
            auto oldObj = diffContext.oldObject();
            auto newObj = diffContext.newObject();
            bool hasOwnGeom = false;

            if (oldObj && !oldObj->geom().isNull()) {
                hasOwnGeom = true;
                diffEnvelopesById[oid].before.expandToInclude(
                        oldObj->geom()->getEnvelopeInternal());
                if (diffContext.geomChanged()) {
                    diffEnvelopesById[oid].removed.expandToInclude(
                            oldObj->geom()->getEnvelopeInternal());
                }
            }
            if (newObj && !newObj->geom().isNull()) {
                hasOwnGeom = true;
                diffEnvelopesById[oid].after.expandToInclude(
                        newObj->geom()->getEnvelopeInternal());
                if (diffContext.geomChanged()) {
                    diffEnvelopesById[oid].added.expandToInclude(
                            newObj->geom()->getEnvelopeInternal());
                }
            }
            if (hasOwnGeom) {
                withOwnGeom.insert(oid);
            } else {
                for (const auto& rel : diffContext.relationsAdded()) {
                    if (rel.masterId == oid && isGeomPartRelation(rel)) {
                        partIdToMastersDiff[rel.slaveId].added.insert(oid);
                        nextIds.insert(rel.slaveId);
                    }
                }
                for (const auto& rel : diffContext.relationsDeleted()) {
                    if (rel.masterId == oid && isGeomPartRelation(rel)) {
                        partIdToMastersDiff[rel.slaveId].removed.insert(rel.masterId);
                        nextIds.insert(rel.slaveId);
                    }
                }

                if (!diffContext.stateChanged()) {
                    toLoadPartIds.insert(oid);
                }
            }
        }

        for (const auto& pair : oldSnapshot.objectRevisions(toLoadRev)) {
            const auto& rev = pair.second;
            if (!rev.data().deleted) {
                if (rev.data().geometry) {
                    withOwnGeom.insert(rev.id().objectId());
                    Geom geom(*rev.data().geometry);
                    diffEnvelopesById[rev.id().objectId()]
                        .before.expandToInclude(geom->getEnvelopeInternal());
                    diffEnvelopesById[rev.id().objectId()]
                        .after.expandToInclude(geom->getEnvelopeInternal());
                } else {
                    toLoadPartIds.insert(rev.id().objectId());
                }
            }
        }

        for (const auto& rev : oldSnapshot.loadSlaveRelations(toLoadPartIds)) {
            auto rel = extractRelation(rev);
            if (isGeomPartRelation(rel)) {
                auto& masters = partIdToMastersDiff[rel.slaveId];
                if (!masters.removed.count(rel.masterId)) {
                    masters.remained.insert(rel.masterId);
                    nextIds.insert(rel.slaveId);
                }
            }
        }

        currentIds = std::move(nextIds);
    }

    // Second, go up geom-part relations and propagate diff envelopes
    // to part masters.
    currentIds = std::move(withOwnGeom);
    while (!currentIds.empty()) {
        std::set<TId> nextIds;
        for (TId oid : currentIds) {
            auto mastersIt = partIdToMastersDiff.find(oid);
            if (mastersIt == partIdToMastersDiff.end()) {
                continue;
            }
            const auto& masters = mastersIt->second;

            const auto& envelopes = diffEnvelopesById[oid];
            for (auto masterId : masters.remained) {
                auto& masterEnvelopes = diffEnvelopesById[masterId];
                masterEnvelopes.before.expandToInclude(&envelopes.before);
                masterEnvelopes.after.expandToInclude(&envelopes.after);
                masterEnvelopes.added.expandToInclude(&envelopes.added);
                masterEnvelopes.removed.expandToInclude(&envelopes.removed);
                nextIds.insert(masterId);
            }

            for (auto masterId : masters.added) {
                auto& masterEnvelopes = diffEnvelopesById[masterId];
                masterEnvelopes.after.expandToInclude(&envelopes.after);
                masterEnvelopes.added.expandToInclude(&envelopes.after);
                nextIds.insert(masterId);
            }

            for (auto masterId : masters.removed) {
                auto& masterEnvelopes = diffEnvelopesById[masterId];
                masterEnvelopes.before.expandToInclude(&envelopes.before);
                masterEnvelopes.removed.expandToInclude(&envelopes.before);
                nextIds.insert(masterId);
            }
        }
        currentIds = std::move(nextIds);
    }

    for (auto& pair : diffEnvelopesById) {
        auto& diffEnvelopes = pair.second;
        fixEnvelope(diffEnvelopes.before);
        fixEnvelope(diffEnvelopes.added);
        fixEnvelope(diffEnvelopes.removed);
        fixEnvelope(diffEnvelopes.after);
    }

    return diffEnvelopesById;
}

} // namespace

DiffEnvelopes
calcObjectDiffEnvelopes(
        TId objectId,
        const std::map<TId, const LongtaskDiffContext*>& diffContextById,
        const revision::Snapshot& oldSnapshot)
{
    return calcDiffEnvelopesById({objectId}, diffContextById, oldSnapshot)[objectId];
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
