#include "extractors.h"
#include "magic_strings.h"
#include <yandex/maps/wiki/diffalert/object.h>

#include <maps/libs/common/include/exception.h>
#include <maps/libs/common/include/map_at.h>
#include <yandex/maps/wiki/revision/objectrevision.h>

#include <boost/algorithm/string/predicate.hpp>
#include <string>

namespace mwr = maps::wiki::revision;

namespace maps {
namespace wiki {
namespace diffalert {

Relation extractRelation(const mwr::ObjectRevision& relRev)
{
    ASSERT(relRev.type() == mwr::RevisionType::Relation);

    const auto& data = relRev.data();
    REQUIRE(data.relationData,
        "Relation " << relRev.id() << " has no relation data");
    REQUIRE(data.attributes, "Relation " << relRev.id() << " has no attributes");
    auto roleIt = data.attributes->find(rel::ROLE);
    REQUIRE(roleIt != std::end(*data.attributes),
        "Relation " << relRev.id() << " has no role");

    return {
        data.relationData->masterObjectId(),
        data.relationData->slaveObjectId(),
        roleIt->second,
        std::stoul(mapAt(*data.attributes, rel::SEQ_NUM, DEFAULT_SEQ_NUM))
    };
}

std::string extractCategoryId(const mwr::Attributes& attrs)
{
    for (const auto& attr : attrs) {
        if (boost::algorithm::starts_with(attr.first, CATEGORY_ATTR_PREFIX) &&
            attr.first != cat::BOUND_JC) {
            //cat:bound_jc is an exception because multiple categories are not supported yet
            return attr.first.substr(CATEGORY_ATTR_PREFIX.size());
        }
    }
    return {};
}

bool isGeomPartCategory(const std::string& categoryId)
{
    for (const auto& suffix : GEOM_PART_CAT_SUFFIXES) {
        if (boost::algorithm::ends_with(categoryId, suffix)) {
            return true;
        }
    }
    return false;
}

bool isGeomPartRelation(const Relation& relation)
{
    for (const auto& role : GEOM_PART_ROLES) {
        if (relation.role == role) {
            return true;
        }
    }
    return false;
}

bool isInterior(const mwr::ObjectRevision& objRev)
{
    const auto& data = objRev.data();
    REQUIRE(data.attributes, "Object " << objRev.id() << " has no attributes");
    for (const auto& attr : *data.attributes) {
        if (boost::algorithm::ends_with(attr.first, IS_INTERIOR_ATTR_SUFFIX)) {
            return true;
        }
    }
    return false;
}

bool isSystemObjectRevision(const mwr::ObjectRevision& objRev)
{
    if (objRev.type() != mwr::RevisionType::RegularObject) {
        return false;
    }

    const auto& data = objRev.data();
    REQUIRE(data.attributes,
         "Object " << objRev.id() << " has no attributes");
    const auto& categoryId = extractCategoryId(*data.attributes);
    for (const auto& suffix : SYSTEM_OBJECT_CAT_SUFFIXES) {
        if (boost::algorithm::ends_with(categoryId, suffix)) {
            return true;
        }
    }
    return false;
}

bool isTableAttrRevision(const mwr::ObjectRevision& relRev)
{
    if (relRev.type() != mwr::RevisionType::Relation) {
        return false;
    }

    const auto& data = relRev.data();
    REQUIRE(data.attributes, "Relation " << relRev.id() << " has no attributes");
    auto slaveCatAttrIt = data.attributes->find(rel::SLAVE);
    REQUIRE(slaveCatAttrIt != std::end(*data.attributes),
        "Couldn't determine relation " << relRev.id() << " slave category");
    for (const auto& suffix : SYSTEM_OBJECT_CAT_SUFFIXES) {
        if (boost::algorithm::ends_with(slaveCatAttrIt->second, suffix)) {
            return true;
        }
    }
    return false;
}

} // namespace diffalert
} // namespace wiki
} // namespace maps
