#include <maps/wikimap/mapspro/libs/editor_client/impl/request.h>

#include "magic_strings.h"

#include <maps/libs/json/include/value.h>
#include <maps/libs/geolib/include/bounding_box.h>
#include <maps/libs/geolib/include/conversion.h>

#include <boost/uuid/uuid.hpp>
#include <boost/uuid/uuid_generators.hpp>
#include <boost/uuid/uuid_io.hpp>

namespace maps::wiki::editor_client {
namespace {
const geolib3::Point2 DEFAULT_CONTEXT_CENTER(37.400283850, 55.794628791);

geolib3::Point2 center(const geolib3::SimpleGeometryVariant& variant)
{
    return variant.visit(
        [](const auto& geometry) {
            return geometry.boundingBox().center();
        });
}

std::string
generateUUID()
{
    std::ostringstream uuidStream;
    uuidStream << boost::uuids::random_generator()();
    return uuidStream.str();
}
} // namespace

std::string createJsonUpdateRequest(
    const BasicEditorObject& object,
    std::vector<DiffAlertMessage> messages)
{
    maps::json::Builder builder;
    builder << [&](maps::json::ObjectBuilder objectBuilder) {
        objectBuilder[json::ID] = std::to_string(object.id);
        objectBuilder[json::CATEGORY_ID] = object.categoryId;
        if (object.id) {
            objectBuilder[json::REVISION_ID] = boost::lexical_cast<std::string>(*object.revisionId);
        }
        objectBuilder[json::UUID] = generateUUID();
        objectBuilder[json::STATE] = json::DRAFT;
        auto geoGeometry = object.getGeometryInGeodetic();
        if (geoGeometry) {
            objectBuilder[json::GEOMETRY] = geolib3::geojson(geoGeometry.value());
        }
        objectBuilder[json::ATTRS] << [&](maps::json::ObjectBuilder attrsBuilder) {
            for (const auto& attr : object.plainAttributes) {
                attrsBuilder[attr.first] = attr.second;
            }
            for (const auto& attr : object.multiValueAttributes) {
                if (object.tableAttributes.contains(attr.first)) {
                    REQUIRE(attr.second.empty(), "Repeated multivalue/table attr: " << attr.first);
                    continue;
                }
                attrsBuilder[attr.first] << [&](maps::json::ArrayBuilder valuesBuilder) {
                    for (const auto& value : attr.second) {
                        valuesBuilder << value;
                    }
                };
            }
            for (const auto& tableAttr : object.tableAttributes) {
                const auto& name = tableAttr.first;
                const auto& rows = tableAttr.second;
                attrsBuilder[name] << [&](maps::json::ArrayBuilder rowsBuilder) {
                    for (const auto& row : rows) {
                        rowsBuilder << [&](maps::json::ObjectBuilder rowBuilder) {
                            for (const auto& column : row) {
                                rowBuilder[column.first] = column.second;
                            }
                        };
                    }
                };
            }
        };
        objectBuilder[json::EDIT_CONTEXT] << [&](maps::json::ObjectBuilder contextBuilder) {
            contextBuilder[json::ZOOM] = 23;
            if (geoGeometry) {
                contextBuilder[json::CENTER] = geolib3::geojson(center(geoGeometry.value()));
            } else {
                contextBuilder[json::CENTER] = geolib3::geojson(DEFAULT_CONTEXT_CENTER);
            }
            if (!messages.empty()) {
                contextBuilder[json::DIFFALERTS] <<
                    [&](maps::json::ArrayBuilder alerts) {
                        for (const auto& message : messages) {
                            alerts << [&](maps::json::ObjectBuilder alert) {
                                alert[json::PRIORITY] = message.priority;
                                alert[json::MESSAGE] = message.message;
                            };
                        }
                    };
            }
        };
    };
    return builder.str();
}

} // maps::wiki::editor_client
