#include <yandex/maps/wiki/filters/stored_expression.h>

#include "parser.h"
#include "db_strings.h"

#include <yandex/maps/wiki/filters/exception.h>


namespace maps::wiki::filters {

StoredExpression::StoredExpression(const pqxx::row& row)
    : id_(row[EXPRESSION_ID_COLUMN].as<TExpressionId>())
    , createdAt_(row[EXPRESSION_CREATED_AT_COLUMN].as<std::string>())
    , createdBy_(row[EXPRESSION_CREATED_BY_COLUMN].as<TUId>())
    , text_(row[EXPRESSION_COLUMN].as<std::string>())
    , langVersion_(row[LANG_VERSION_COLUMN].as<std::string>())
{ }


StoredExpression::~StoredExpression() { }

TExpressionId StoredExpression::id() const { return id_; }
const std::string& StoredExpression::createdAt() const { return createdAt_; }
TUId StoredExpression::createdBy() const { return createdBy_; }

const std::string& StoredExpression::text() const { return text_; }
const std::string& StoredExpression::langVersion() const { return langVersion_; }

ParsedExpression StoredExpression::parsed() const
{
    if (langVersion_ != CURRENT_LANG_VERSION) {
        throw ParseError()
            << "expression lang version: " << langVersion_
            << " differs from current lang version: " << CURRENT_LANG_VERSION;
    }
    return ParsedExpression::parse(text_);
}


StoredExpression StoredExpression::create(
        pqxx::transaction_base& txn, TUId uid, const std::string& text)
{
    //check syntax
    ParsedExpression::parse(text);

    const auto query =
        "INSERT INTO "  + FILTER_EXPRESSION_TABLE
        + "("
        + EXPRESSION_CREATED_BY_COLUMN
        + "," + LANG_VERSION_COLUMN
        + "," + EXPRESSION_COLUMN
        + ")"
        + " VALUES($1,$2,$3)"
        + " RETURNING " + STORED_EXPRESSION_COLUMNS;

    pqxx::result r = txn.exec_params(query, uid, CURRENT_LANG_VERSION, text);
    return StoredExpression(r.at(0));
}

StoredExpression StoredExpression::load(pqxx::transaction_base& txn, TExpressionId id)
{
    const auto query =
        "SELECT " + STORED_EXPRESSION_COLUMNS
        + " FROM " + FILTER_EXPRESSION_TABLE
        + " WHERE " + EXPRESSION_ID_COLUMN + "=" + std::to_string(id);

    pqxx::result r = txn.exec(query);
    if (r.empty()) {
        throw ExpressionNotFound() << "expression id " << id << " not found";
    }
    return StoredExpression(r[0]);
}

} // maps::wiki::filters
