#include <yandex/maps/wiki/filters/stored_filter.h>
#include <yandex/maps/wiki/filters/exception.h>

#include "db_strings.h"
#include "parser.h"

namespace maps::wiki::filters {

namespace {

const std::string LESS = "<";
const std::string GREATER = ">";

const std::string ASC = "ASC";
const std::string DESC = "DESC";

} // namespace

void StoredFilter::initAttrs(const pqxx::row& row)
{
    id_ = row[FILTER_ID_COLUMN].as<TFilterId>();
    createdAt_ = row[CREATED_AT_COLUMN].as<std::string>();
    createdBy_ = row[CREATED_BY_COLUMN].as<TUId>();
    isDeleted_ = row[IS_DELETED_COLUMN].as<bool>();
    name_ = row[NAME_COLUMN].as<std::string>();
    isPublic_ = row[IS_PUBLIC_COLUMN].as<bool>();
}

StoredFilter::StoredFilter(const pqxx::row& row, StoredExpression expression)
    : expression_(std::move(expression))
{
    initAttrs(row);
}

StoredFilter::StoredFilter(const pqxx::row& joinedRow)
    : expression_(joinedRow)
{
    initAttrs(joinedRow);
}


StoredFilter::~StoredFilter() { }


TFilterId StoredFilter::id() const
{
   return id_;
}

const std::string& StoredFilter::createdAt() const
{ return createdAt_; }

TUId StoredFilter::createdBy() const
{ return createdBy_; }

bool StoredFilter::isDeleted() const
{ return isDeleted_; }

void StoredFilter::setDeleted(pqxx::transaction_base& txn)
{
    const auto query =
        "UPDATE "  + FILTER_TABLE
        + " SET " + IS_DELETED_COLUMN + "=TRUE"
        + " WHERE " + FILTER_ID_COLUMN + "=" + std::to_string(id_);

    txn.exec(query);
    isDeleted_ = true;
}

const std::string& StoredFilter::name() const
{ return name_; }

void StoredFilter::setName(pqxx::transaction_base& txn, const std::string& value)
{
    if (name_ == value) {
        return;
    }

    const auto query =
        "UPDATE "  + FILTER_TABLE
        + " SET " + NAME_COLUMN + "=" + txn.quote(value)
        + " WHERE " + FILTER_ID_COLUMN + "=" + std::to_string(id_);

    txn.exec(query);
    name_ = value;
}

bool StoredFilter::isPublic() const
{ return isPublic_; }

void StoredFilter::setPublic(pqxx::transaction_base& txn, bool value)
{
    if (isPublic_ == value) {
        return;
    }

    const auto query =
        "UPDATE " + FILTER_TABLE
        + " SET " + IS_PUBLIC_COLUMN + (value ? "=TRUE" : "=FALSE")
        + " WHERE " + FILTER_ID_COLUMN + "=" + std::to_string(id_);

    txn.exec(query);
    isPublic_ = value;
}

const StoredExpression& StoredFilter::expression() const
{ return expression_; }

void StoredFilter::setExpression(
    pqxx::transaction_base& txn,
    TUId uid,
    const std::string& value)
{
    if (expression_.langVersion() == CURRENT_LANG_VERSION
            && expression_.text() == value) {
        return;
    }

    auto expr = StoredExpression::create(txn, uid, value);

    const auto query =
        "UPDATE "  + FILTER_TABLE
        + " SET " + EXPRESSION_ID_COLUMN + "=" + std::to_string(expr.id())
        + " WHERE " + FILTER_ID_COLUMN + "=" + std::to_string(id_);

    txn.exec(query);
    expression_ = std::move(expr);
}


StoredFilter StoredFilter::load(pqxx::transaction_base& txn, TFilterId id)
{
    const auto query =
        "SELECT " + STORED_FILTER_COLUMNS
        + " FROM " + FILTER_TABLE + JOIN_FILTER_EXPRESSION_TABLE
        + " WHERE " + FILTER_ID_COLUMN + "=" + std::to_string(id)
        + " AND " + IS_DELETED_COLUMN + "=FALSE";

    pqxx::result r = txn.exec(query);
    if (r.empty()) {
        throw FilterNotFound() << "filter id " << id << " not found";
    }

    return StoredFilter(r[0]);
}

StoredFilter StoredFilter::create(
    pqxx::transaction_base& txn,
    TUId uid,
    const std::string& name,
    bool isPublic,
    const std::string& expression)
{
    auto expr = StoredExpression::create(txn, uid, expression);

    const auto query =
        "INSERT INTO "  + FILTER_TABLE
        + "(" + CREATED_BY_COLUMN
        + "," + CREATED_AT_COLUMN
        + "," + NAME_COLUMN
        + "," + IS_PUBLIC_COLUMN
        + "," + EXPRESSION_ID_COLUMN + ")"
        + " VALUES($1,$2,$3,$4,$5)"
        + " RETURNING " + FILTER_META_COLUMNS;

    pqxx::result r = txn.exec_params(
        query,
        uid, expr.createdAt(), name, isPublic, expr.id());
    return StoredFilter(r.at(0), std::move(expr));
}

template<typename RowIter>
std::vector<StoredFilter>
readFilters(RowIter begin, RowIter end, size_t limit)
{
    std::vector<StoredFilter> filters;
    for (auto it = begin; it != end; ++it) {
        const auto& row = *it;
        filters.emplace_back(row);
        if (limit && filters.size() == limit) {
            break;
        }
    }
    return filters;
}

common::PagedResult<std::vector<StoredFilter>>
StoredFilter::loadPaged(
    pqxx::transaction_base& txn,
    TUId uid,
    size_t page,
    size_t perPage,
    boost::optional<TUId> createdBy)
{
    std::stringstream whereClause;
    whereClause
        << " WHERE ("
        << IS_PUBLIC_COLUMN << "=TRUE"
        << " OR " << CREATED_BY_COLUMN << "=" << uid << ")"
        << " AND " << IS_DELETED_COLUMN << "=FALSE";
    if (createdBy) {
        whereClause << " AND " << CREATED_BY_COLUMN << "=" << *createdBy;
    }

    std::string countQuery =
        "SELECT COUNT(1) FROM " + FILTER_TABLE + whereClause.str();
    size_t count = txn.exec(countQuery).at(0).at(0).as<size_t>();

    common::Pager pager(count, page, perPage);

    std::stringstream loadQuery;
    loadQuery
        << "SELECT " << STORED_FILTER_COLUMNS
        << " FROM " << FILTER_TABLE << JOIN_FILTER_EXPRESSION_TABLE
        << whereClause.str()
        << " ORDER BY " << FILTER_ID_COLUMN << " DESC"
        << " LIMIT " << pager.limit() << " OFFSET " << pager.offset();

    auto rows = txn.exec(loadQuery.str());
    auto filters = readFilters(rows.begin(), rows.end(), pager.limit());

    return common::PagedResult<std::vector<StoredFilter>>(pager, std::move(filters));
}

StoredFilter::ResultBeforeAfter
StoredFilter::load(
    pqxx::transaction_base& txn,
    TUId uid,
    boost::optional<TUId> createdBy,
    const std::string& namePart,
    TFilterId startID,
    common::BeforeAfter beforeAfter,
    size_t limit)
{
    std::string whereClause =
        " WHERE ("
        + IS_PUBLIC_COLUMN + " = TRUE"
        + " OR " + CREATED_BY_COLUMN + " = " + std::to_string(uid) + ")"
        + " AND " + IS_DELETED_COLUMN + " = FALSE";
    if (!namePart.empty()) {
        whereClause += " AND name ILIKE " + txn.quote("%" + txn.esc_like(namePart) + "%");
    }
    if (createdBy) {
        whereClause += " AND " + CREATED_BY_COLUMN + " = " + std::to_string(*createdBy);
    }
    if (startID) {
        const auto startFilter = load(txn, startID);
        whereClause +=
            " AND name "
            + (beforeAfter == common::BeforeAfter::Before ? LESS : GREATER)
            + txn.quote(startFilter.name());
    }

    bool reverse = startID && beforeAfter == common::BeforeAfter::Before;

    std::string query =
        "SELECT " + STORED_FILTER_COLUMNS +
        " FROM " + FILTER_TABLE + JOIN_FILTER_EXPRESSION_TABLE
        + whereClause
        + " ORDER BY name " + (reverse ? DESC : ASC);
    if (limit) {
        query += " LIMIT " + std::to_string(limit + 1);
    }

    const auto rows = txn.exec(query);
    if (rows.empty()) {
        return {};
    }
    const bool hasMore = limit ? (rows.size() > limit) : false;

    return ResultBeforeAfter {
        reverse
            ? readFilters(limit >= rows.size() ? rows.rbegin() : rows.rbegin() + 1, rows.rend(), limit)
            : readFilters(rows.begin(), rows.end(), limit),
        hasMore
    };
}

} // namespace maps::wiki::filters
