#include <maps/wikimap/mapspro/libs/flat_range/include/range.h>

#include "helpers.h"
#include "string_helpers.h"

#include <yandex/maps/wiki/common/string_utils.h>

#include <algorithm>
#include <cctype>
#include <locale>

namespace maps::wiki::flat_range {

namespace {
const size_t MAX_PLAUSIBLE_RANGE_SIZE = 15000;
} // namespace

Range::Range(const std::string& singleValue)
    : constPart_(singleValue)
{
    CHECK(!singleValue.empty(), "Empty value in range");
}

Range::Range(const std::string& start, const std::string& end)
{
    CHECK(!start.empty() || !end.empty(), "Empty bounds in range: " << start << "-" << end);
    if (start == end) {
        constPart_ = start;
        return;
    }
    auto commonPrefixLen = commonNotNumberPrefixLen(start, end);
    if (commonPrefixLen) {
        auto startSuffix = start.substr(commonPrefixLen);
        CHECK(isNumber(startSuffix), "Range start can't be determined: " << start << "-" << end);
        auto endSuffix = end.substr(commonPrefixLen);
        CHECK(isNumber(endSuffix), "Range end can't be determined: " << start << "-" << end);
        constPart_ = start.substr(0, commonPrefixLen);
        numberFirst_ = false;
        start_ = std::stoi(startSuffix);
        end_ = std::stoi(endSuffix);
        return;
    }
    auto commonSuffixLen = commonNotNumberSuffixLen(start, end);
    if (commonSuffixLen) {
        auto startPrefix = start.substr(0, start.length() - commonSuffixLen);
        CHECK(isNumber(startPrefix), "Range start can't be determined: " << start << "-" << end);
        auto endPrefix = end.substr(0, end.length() - commonSuffixLen);
        CHECK(isNumber(endPrefix), "Range end can't be determined: " << start << "-" << end);
        constPart_ = start.substr(start.length() - commonSuffixLen);
        numberFirst_ = true;
        start_ = std::stoi(startPrefix);
        end_ = std::stoi(endPrefix);
        return;
    }
    CHECK(isNumber(start) && isNumber(end), "Range bounds are not numbers: " << start << "-" << end);
    start_ = std::stoi(start);
    end_ = std::stoi(end);
}

size_t
Range::size() const
{
    if (!end_ || !start_) {
        return 1;
    }
    return abs(*end_ - *start_) + 1;
}

std::string
Range::value(size_t n) const
{
    ASSERT(n < size());
    if (!end_ || !start_) {
        return constPart_;
    }
    int value = *start_ + ((*end_ > *start_) ? n : -n);
    if (numberFirst_) {
        return std::to_string(value) + constPart_;
    } else {
        return constPart_ + std::to_string(value);
    }
}

std::string
Range::min() const
{
    if (!end_ || !start_) {
        return constPart_;
    }
    return *end_ > *start_ ? first() : last();
}

std::string
Range::max() const
{
    if (!end_ || !start_) {
        return constPart_;
    }
    return *end_ > *start_ ? last() : first();
}

Ranges parse(const std::string& textInput)
{
    Ranges result;
    auto rangesStrings = splitQuoted(textInput);
    for (auto& rangeString : rangesStrings) {
        if (isQuoted(rangeString)) {
            result.push_back(rangeString);
            continue;
        }
        CHECK(!rangeString.empty(), "Empty value in ranges: " << textInput);
        replaceMinusWithBoundsDelimiterKeepNegative(rangeString);
        auto bounds = common::split(rangeString, BOUNDS_DELIMITER);
        if (bounds.size() == 1) {
            result.emplace_back(bounds[0]);
        } else if (bounds.size() == 2) {
            result.emplace_back(bounds[0], bounds[1]);
        } else {
            CHECK(false, "Malformed range in: " << textInput);
        }
        CHECK(result.back().size() < MAX_PLAUSIBLE_RANGE_SIZE, "Implausible range size: " << textInput);
    }
    return result;
}

} // namespace maps::wiki::flat_range
