#include <yandex/maps/wiki/geolocks/geolocks.h>

#include <yandex/maps/wiki/revision/branch.h>
#include <yandex/maps/wiki/revision/branch_manager.h>
#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/libs/common/include/exception.h>

#include <algorithm>

namespace maps::wiki::geolocks {

namespace rev = revision;

namespace {

const revision::Branch::LockId ADVISORY_LOCK_ID = 3;


const std::string GEOMETRY_LOCK_TABLE = "geolocks.geometry_lock";

const std::string ID_COLUMN = "id";
const std::string CREATED_BY_COLUMN = "created_by";
const std::string CREATED_AT_COLUMN = "created_at";

const std::string BRANCH_ID_COLUMN = "branch_id";
const std::string COMMIT_ID_COLUMN = "commit_id";
const std::string EXTENT_COLUMN = "extent";

const std::string GEOLOCK_COLUMNS =
    ID_COLUMN
    + "," + CREATED_BY_COLUMN + "," + CREATED_AT_COLUMN
    + "," + BRANCH_ID_COLUMN + "," + COMMIT_ID_COLUMN
    + ",ST_AsBinary(" + EXTENT_COLUMN + ") as " + EXTENT_COLUMN;

const std::string WKB_POLYGON_HEADER_LE("\x01\x03\x00\x00\x00", 5);
const std::string WKB_POLYGON_HEADER_BE("\x00\x00\x00\x00\x03", 5);

const std::string LOAD_GEOLOCK =
    "SELECT "
    + GEOLOCK_COLUMNS
    + " FROM " + GEOMETRY_LOCK_TABLE
    + " WHERE "
    + BRANCH_ID_COLUMN + "=$1 AND ";
const std::string INTERSECTS_BBOX_CONDITION =
    "ST_Intersects(" + EXTENT_COLUMN + ", ST_MakeBox2D(ST_MakePoint($2,$3),ST_MakePoint($4,$5)))";
const std::string CONTAINS_POINT_CONDITION =
    "ST_Contains(" + EXTENT_COLUMN + ", ST_MakePoint($2,$3))";

const std::string GEOLOCK_LOAD_BY_POINT_PREPARED_CONTENTS =
    "SELECT "
    + GEOLOCK_COLUMNS
    + " FROM " + GEOMETRY_LOCK_TABLE
    + " WHERE "
    + BRANCH_ID_COLUMN + "=$1"
    + " AND ST_Contains("
            + EXTENT_COLUMN
            + ", ST_MakePoint($2,$3))";

const std::string GEOLOCK_TRY_LOCK_QUERY =
    std::string(" WITH to_insert AS (")
    + "SELECT $1::bigint, $2::bigint, $3::bigint, $4::geometry WHERE NOT EXISTS ("
    + " SELECT * FROM " + GEOMETRY_LOCK_TABLE + " WHERE "
    + BRANCH_ID_COLUMN + "=$2"
    + " AND ST_Intersects(" + EXTENT_COLUMN + ",$4))"
    + ")"
    + " INSERT INTO " + GEOMETRY_LOCK_TABLE
    + "("  + CREATED_BY_COLUMN
    + "," + BRANCH_ID_COLUMN + "," + COMMIT_ID_COLUMN
    + "," + EXTENT_COLUMN + ")"
    + " (SELECT * FROM to_insert)"
    + " RETURNING " + ID_COLUMN + "," + CREATED_AT_COLUMN;

revision::DBID lockBranch(pqxx::transaction_base& txn, revision::DBID branchId)
{
    auto branch = rev::BranchManager(txn).load(branchId);
    branch.lock(
            txn, ADVISORY_LOCK_ID,
            rev::Branch::LockMode::Wait, rev::Branch::LockType::Exclusive);

    return revision::RevisionsGateway(txn, branch).headCommitId();
}

bool isPolygonWkb(const std::string& wkb)
{
    return wkb.starts_with(WKB_POLYGON_HEADER_LE)
        || wkb.starts_with(WKB_POLYGON_HEADER_BE);
}

std::string loadGeolocksQuery(
    const std::string& geometryCondition,
    GeolockType geolockType)
{
    switch (geolockType) {
        case GeolockType::Manual:
            return LOAD_GEOLOCK + geometryCondition + " AND " + COMMIT_ID_COLUMN + " = 0";
        case GeolockType::Auto:
            return LOAD_GEOLOCK + geometryCondition + " AND " + COMMIT_ID_COLUMN + " != 0";
        case GeolockType::All:
            return LOAD_GEOLOCK + geometryCondition;
    }
}

bool isLocked(
        pqxx::transaction_base& txn,
        revision::DBID branchId,
        const std::string& geometryCondition,
        std::function<pqxx::result(const std::string&)> checkGeolock,
        GeolockType geolockType)
{
    rev::BranchManager(txn).load(branchId).lock(
        txn,
        ADVISORY_LOCK_ID,
        rev::Branch::LockMode::Wait,
        rev::Branch::LockType::Shared);

    std::string loadLocks = loadGeolocksQuery(geometryCondition, geolockType);
    pqxx::result r = checkGeolock("SELECT EXISTS (" + loadLocks + ")");
    return r.front().front().as<bool>();
}

} // namespace

class GeoLock::Impl
{
public:
    Impl(const pqxx::row& row)
        : id(row[ID_COLUMN].as<TLockId>())
        , createdBy(row[CREATED_BY_COLUMN].as<TUId>())
        , createdAt(row[CREATED_AT_COLUMN].as<std::string>())
        , branchId(row[BRANCH_ID_COLUMN].as<revision::DBID>())
        , commitId(row[COMMIT_ID_COLUMN].as<revision::DBID>())
        , extentWkb(pqxx::binarystring(row[EXTENT_COLUMN]).str())
    { }

    Impl(
            const pqxx::row& insertedRow,
            TUId createdBy_,
            revision::DBID branchId_, revision::DBID commitId_, std::string extentWkb_)
        : id(insertedRow[ID_COLUMN].as<TLockId>())
        , createdBy(createdBy_)
        , createdAt(insertedRow[CREATED_AT_COLUMN].as<std::string>())
        , branchId(branchId_)
        , commitId(commitId_)
        , extentWkb(std::move(extentWkb_))
    { }

    static std::unique_ptr<GeoLock::Impl> tryLock(
            pqxx::transaction_base& txn, TUId uid,
            revision::DBID branchId, revision::DBID commitId,
            const std::string& extentWkb)
    {
        auto r = txn.exec_params(GEOLOCK_TRY_LOCK_QUERY,
            uid, branchId, commitId, pqxx::binarystring(extentWkb));

        ASSERT(r.size() <= 1);
        std::unique_ptr<Impl> implPtr;
        if (!r.empty()) {
            implPtr.reset(
                new Impl(r[0], uid, branchId, commitId, std::move(extentWkb)));
        }
        return implPtr;
    }

    TLockId id;
    TUId createdBy;
    std::string createdAt;

    revision::DBID branchId;
    revision::DBID commitId;
    std::string extentWkb;
};

GeoLock::GeoLock(std::unique_ptr<Impl>&& impl)
    : impl_(std::move(impl))
{ }

GeoLock::~GeoLock() = default;

GeoLock::GeoLock(const GeoLock& other)
    : impl_(std::unique_ptr<Impl>(new Impl(*other.impl_)))
{ }

GeoLock::GeoLock(GeoLock&& other) noexcept
    : impl_(std::move(other.impl_))
{ }

GeoLock& GeoLock::operator=(const GeoLock& other)
{
    if (&other != this) {
        GeoLock temp(other);
        impl_.swap(temp.impl_);
    }
    return *this;
}

GeoLock& GeoLock::operator=(GeoLock&& other) noexcept
{
    impl_ = std::move(other.impl_);
    return *this;
}

TLockId GeoLock::id() const { return impl_->id; }
TUId GeoLock::createdBy() const { return impl_->createdBy; }
const std::string& GeoLock::createdAt() const { return impl_->createdAt; }

revision::DBID GeoLock::branchId() const { return impl_->branchId; }
revision::DBID GeoLock::commitId() const { return impl_->commitId; }
const std::string& GeoLock::extentWkb() const { return impl_->extentWkb; }


GeoLock GeoLock::load(pqxx::transaction_base& txn, TLockId id)
{
    const auto query =
        "SELECT " + GEOLOCK_COLUMNS
        + " FROM " + GEOMETRY_LOCK_TABLE
        + " WHERE "
        + ID_COLUMN + "=" + std::to_string(id);

    pqxx::result r = txn.exec(query);
    if (r.empty()) {
        throw NotFoundException() << "lock id: " << id << " not found";
    }
    return GeoLock{std::unique_ptr<Impl>(new Impl(r[0]))};
}

std::list<GeoLock> GeoLock::loadByGeom(
        pqxx::transaction_base& txn,
        revision::DBID branchId,
        const geolib3::BoundingBox& bbox,
        GeolockType geolockType)
{
    const auto& query = loadGeolocksQuery(INTERSECTS_BBOX_CONDITION, geolockType);
    pqxx::result r = txn.exec_params(
        query, branchId, bbox.minX(), bbox.minY(), bbox.maxX(), bbox.maxY());
    std::list<GeoLock> locks;
    for (const auto& row : r) {
        locks.push_back(GeoLock(std::unique_ptr<Impl>(new Impl(row))));
    }

    return locks;
}

std::list<GeoLock> GeoLock::loadByGeom(
        pqxx::transaction_base& txn,
        revision::DBID branchId,
        const geolib3::Point2& point,
        GeolockType geolockType)
{
    const auto& query = loadGeolocksQuery(CONTAINS_POINT_CONDITION, geolockType);
    pqxx::result r = txn.exec_params(query, branchId, point.x(), point.y());
    std::list<GeoLock> locks;
    for (const auto& row : r) {
        locks.push_back(GeoLock(std::unique_ptr<Impl>(new Impl(row))));
    }

    return locks;
}

boost::optional<GeoLock> GeoLock::tryLock(
        pqxx::connection_base& conn, TUId uid,
        revision::DBID branchId, const std::string& extentWkb)
{
    REQUIRE(isPolygonWkb(extentWkb), "geolock extent should be polygonal");

    pqxx::work txn(conn);
    auto headCommitId = lockBranch(txn, branchId);

    auto lockImplPtr =
        Impl::tryLock(txn, uid, branchId, headCommitId, extentWkb);
    if (!lockImplPtr) {
        return {};
    }

    txn.commit();
    return GeoLock(std::move(lockImplPtr));
}

boost::optional<std::vector<GeoLock>> GeoLock::tryLockAll(
        pqxx::connection_base& conn, TUId uid,
        revision::DBID branchId,
        const std::vector<std::string>& extentWkbs)
{
    if (extentWkbs.empty()) {
        return std::vector<GeoLock>();
    }
    REQUIRE(std::all_of(
            std::begin(extentWkbs), std::end(extentWkbs), isPolygonWkb),
        "geolock extents should be polygonal");

    pqxx::work txn(conn);
    auto headCommitId = lockBranch(txn, branchId);

    std::vector<GeoLock> locks;
    locks.reserve(extentWkbs.size());
    for (auto& extentWkb : extentWkbs) {
        auto lockImplPtr =
            Impl::tryLock(txn, uid, branchId, headCommitId, extentWkb);
        if (!lockImplPtr) {
            return {};
        }
        locks.push_back(GeoLock(std::move(lockImplPtr)));
    }

    txn.commit();
    return locks;
}

void GeoLock::unlock(pqxx::transaction_base& txn)
{
    const auto query =
        "DELETE FROM " + GEOMETRY_LOCK_TABLE
        + " WHERE " + ID_COLUMN + "=" + std::to_string(id());

    pqxx::result r = txn.exec(query);
    if (r.affected_rows() == 0) {
        throw NotFoundException() << "lock id: " << id() << " not found";
    }
}

void GeoLock::unlock(pqxx::connection_base& conn)
{
    pqxx::work txn(conn);
    unlock(txn);
    txn.commit();
}

bool isLocked(
        pqxx::transaction_base& txn,
        revision::DBID branchId,
        const geolib3::BoundingBox& bbox,
        GeolockType geolockType)
{
    return isLocked(
        txn,
        branchId,
        INTERSECTS_BBOX_CONDITION,
        [&txn, &bbox, &branchId](const std::string& query)
        {
            return txn.exec_params(
                query, branchId, bbox.minX(), bbox.minY(), bbox.maxX(), bbox.maxY());
        },
        geolockType
    );
}

bool isLocked(
        pqxx::transaction_base& txn,
        revision::DBID branchId,
        const geolib3::Point2& point,
        GeolockType geolockType)
{
    return isLocked(
        txn,
        branchId,
        CONTAINS_POINT_CONDITION,
        [&txn, &point, &branchId](const std::string& query)
        {
            return txn.exec_params(query, branchId, point.x(), point.y());
        },
        geolockType
    );
}

} // namespace maps::wiki::geolocks
