#include "cut.h"

#include "cut_line.h"
#include "cutter.h"

#include <queue>
#include <array>
#include <initializer_list>
#include <map>

namespace maps {
namespace wiki {
namespace geom_tools {

namespace {

typedef std::vector<double> CoordsVector;

CoordsVector
sortedCoordinates(const GeolibPolygonVector& polygons, Direction direction)
{
    size_t totalPointsCount = 0;
    for (const geolib3::Polygon2& poly : polygons) {
        totalPointsCount += poly.totalPointsNumber();
    }

    CoordsVector coords;
    coords.reserve(totalPointsCount);

    auto collectCoords = [&] (const geolib3::LinearRing2& ring)
    {
        for (size_t i = 0; i < ring.pointsNumber(); ++i) {
            const auto& point = ring.pointAt(i);
            coords.push_back(direction == Direction::X ? point.x() : point.y());
        }
    };

    for (const geolib3::Polygon2& poly : polygons) {
        collectCoords(poly.exteriorRing());
        for (size_t hIdx = 0; hIdx < poly.interiorRingsNumber(); ++hIdx) {
            collectCoords(poly.interiorRingAt(hIdx));
        }
    }

    std::sort(coords.begin(), coords.end());

    return coords;
}

/// stop cutting at this polygon width or height (corresponding to cut direction)

struct CoordsIndexRange {
    size_t min;
    size_t max;
};

struct SplitPolygonsTask {
    GeolibPolygonVector polygons;
    CoordsIndexRange xRange;
    CoordsIndexRange yRange;
};

typedef std::queue<SplitPolygonsTask> SplitTasksQueue;

struct SplitContext {
    SplitTasksQueue queue;
    CoordsVector xCoords;
    CoordsVector yCoords;
    size_t maxVertices;
    double minSize;
    double tolerance;
    GeolibPolygonVector result;
};

bool
needsCut(const geolib3::Polygon2& poly, size_t maxVertices, double minSize)
{
    if (poly.totalPointsNumber() <= maxVertices) {
        return false;
    }
    const auto bbox = poly.boundingBox();
    return !(bbox.width() < minSize && bbox.height() < minSize);
}

struct SubtaskParams {
    Direction direction;
    size_t cutCoordIndex;
};

GeolibPolygonVector
processSubTask(const SplitPolygonsTask& task, SplitContext& context, const SubtaskParams& params)
{
    const auto& range = params.direction == Direction::X ? task.xRange : task.yRange;
    const auto& coords = params.direction == Direction::X ? context.xCoords : context.yCoords;
    CoordsIndexRange newLessRange = {range.min, params.cutCoordIndex};
    CoordsIndexRange newGreaterRange = {params.cutCoordIndex, range.max};

    PolygonCutter cutter(
        CutLine(params.direction, coords[params.cutCoordIndex], context.tolerance));
    GeolibPolygonVector uncutPolygons, lessPolygons, greaterPolygons;
    for (const auto& polygon : task.polygons) {
        auto res = cutter(polygon);
        if (!res) {
            uncutPolygons.push_back(polygon);
            continue;
        }
        for (auto& poly : res->less) {
            (needsCut(poly, context.maxVertices, context.minSize)
                ? lessPolygons
                : context.result).push_back(std::move(poly));
        }
        for (auto& poly : res->greater) {
            (needsCut(poly, context.maxVertices, context.minSize)
                ? greaterPolygons
                : context.result).push_back(std::move(poly));
        }
    }

    if (!lessPolygons.empty()) {
        context.queue.push(SplitPolygonsTask{
            std::move(lessPolygons),
            params.direction == Direction::X ? newLessRange : task.xRange,
            params.direction == Direction::Y ? newLessRange : task.yRange});
    }
    if (!greaterPolygons.empty()) {
        context.queue.push(SplitPolygonsTask{
            std::move(greaterPolygons),
            params.direction == Direction::X ? newGreaterRange : task.xRange,
            params.direction == Direction::Y ? newGreaterRange : task.yRange});
    }

    return uncutPolygons;
}

typedef std::array<size_t, 2> Weights;

void
processTask(SplitPolygonsTask task, SplitContext& context)
{
    std::map<double, SubtaskParams> subtasks;
    for (Direction d : {Direction::X, Direction::Y}) {
        for (Weights weights : { Weights{{1, 1}}, Weights{{1, 2}}, Weights{{2, 1}} }) {
            const auto& range = d == Direction::X ? task.xRange : task.yRange;
            const auto& coords = d == Direction::X ? context.xCoords : context.yCoords;
            auto cutCoordIndex = (weights[0] * range.min + weights[1] * range.max) /
                (weights[0] + weights[1]);
            CoordsIndexRange newLessRange = {range.min, cutCoordIndex};
            CoordsIndexRange newGreaterRange = {cutCoordIndex, range.max};
            if (newLessRange.min >= newLessRange.max || newGreaterRange.min >= newGreaterRange.max) {
                continue;
            }
            const double lessSize = coords[newLessRange.max] - coords[newLessRange.min];
            const double greaterSize = coords[newGreaterRange.max] - coords[newGreaterRange.min];
            if (lessSize < context.minSize || greaterSize < context.minSize) {
                continue;
            }
            subtasks.insert({std::min(lessSize, greaterSize), SubtaskParams{d, cutCoordIndex}});
        }
    }

    if (subtasks.empty()) {
        // current dimensions too small
        context.result.insert(context.result.end(), task.polygons.begin(), task.polygons.end());
        return;
    }

    auto subtaskIt = subtasks.begin();
    while (!task.polygons.empty()) {
        REQUIRE(subtaskIt != subtasks.end(), "No attempts left to cut polygons");
        task.polygons = processSubTask(task, context, subtaskIt->second);
        ++subtaskIt;
    }
}

} // namespace

GeolibPolygonVector
cut(
    const GeolibPolygonVector& polygons,
    size_t maxVertices,
    double minSize,
    double tolerance)
{
    GeolibPolygonVector result;
    GeolibPolygonVector uncutPolygons;
    for (const geolib3::Polygon2& polygon : polygons) {
        (needsCut(polygon, maxVertices, minSize) ? uncutPolygons : result).push_back(polygon);
    }
    if (uncutPolygons.empty()) {
        return result;
    }

    SplitContext context = {
        SplitTasksQueue(),
        sortedCoordinates(uncutPolygons, Direction::X),
        sortedCoordinates(uncutPolygons, Direction::Y),
        maxVertices,
        minSize,
        tolerance,
        std::move(result)
    };

    context.queue.push(
        SplitPolygonsTask{
            std::move(uncutPolygons),
            {0, context.xCoords.size() - 1},
            {0, context.yCoords.size() - 1}});

    while (!context.queue.empty()) {
        processTask(std::move(context.queue.front()), context);
        context.queue.pop();
    }

    return context.result;
}

} // namespace geom_tools
} // namespace wiki
} // namespace maps
