#include "polygonal_adapters.h"

#include <maps/libs/geolib/include/spatial_relation.h>

#include <list>

namespace maps {
namespace wiki {
namespace geom_tools {

// RingAdapter

void
RingAdapter::checkPointNumber(size_t number) const
{
    const size_t npoints = pointsCount();
    REQUIRE(npoints >= 3, "Invalid points count in ring");
    if (number >= npoints) {
        throw OutOfRangeException() << "Point number out of range: " << number;
    }
}

void
RingAdapter::cacheArea() const
{
    const size_t npoints = pointsCount();
    REQUIRE(npoints >= 3, "Invalid points count in ring");

    double sum = 0.0;
    for (size_t i = 0; i < npoints; ++i) {
        size_t j = (i < npoints - 1) ? i + 1 : 0;
        sum += (ring().pointAt(j).x() + ring().pointAt(i).x()) *
            (ring().pointAt(j).y() - ring().pointAt(i).y());
    }
    area_ = sum / 2.0;
}

void
RingAdapter::cacheBBox() const
{
    const size_t npoints = pointsCount();
    REQUIRE(npoints >= 3, "Invalid points count in ring");

    double minx, maxx, miny, maxy;
    minx = maxx = ring().pointAt(0).x();
    miny = maxy = ring().pointAt(0).y();
    for (size_t i = 1; i < npoints; ++i) {
        const auto& point = ring().pointAt(i);
        minx = std::min(minx, point.x());
        maxx = std::max(maxx, point.x());
        miny = std::min(miny, point.y());
        maxy = std::max(maxy, point.y());
    }
    bbox_ = geolib3::BoundingBox({minx, miny}, {maxx, maxy});
}

void
RingAdapter::cacheOrientation() const
{
    actualOrientation_ = signedAreaImpl() < 0 ? Orientation::CW : Orientation::CCW;
}

geolib3::Point2
RingAdapter::point(size_t number) const
{
    checkPointNumber(number);
    return !requestedOrientation_ || *requestedOrientation_ == actualOrientation()
        ? ring().pointAt(number)
        : ring().pointAt(pointsCount() - 1 - number);
}

geolib3::Point2
RingAdapter::prevPoint(size_t number) const
{
    return point(prevPointNumber(number));
}

geolib3::Point2
RingAdapter::nextPoint(size_t number) const
{
    return point(nextPointNumber(number));
}

geolib3::Segment2
RingAdapter::segment(size_t number) const
{
    return {point(number), nextPoint(number)};
}

geolib3::Segment2
RingAdapter::prevSegment(size_t number) const
{
    return {prevPoint(number), point(number)};
}

geolib3::Segment2
RingAdapter::nextSegment(size_t number) const
{
    size_t nextNum = nextPointNumber(number);
    return {point(nextNum), nextPoint(nextNum)};
}

geolib3::BoundingBox
RingAdapter::boundingBox() const
{
    if (!bbox_) {
        cacheBBox();
    }

    return *bbox_;
}

double
RingAdapter::signedAreaImpl() const
{
    if (!area_) {
        cacheArea();
    }

    return *area_;
}

double
RingAdapter::signedArea() const
{
    auto area = std::fabs(signedAreaImpl());
    return orientation() == Orientation::CCW ? area : -area;
}

Orientation
RingAdapter::orientation() const
{
    if (requestedOrientation_) {
        return *requestedOrientation_;
    }
    return actualOrientation();
}

Orientation
RingAdapter::actualOrientation() const
{
    if (!actualOrientation_) {
        cacheOrientation();
    }
    return *actualOrientation_;
}

void
RingAdapter::setOrientation(Orientation requiredOrientation)
{
    requestedOrientation_ = requiredOrientation;
}


RingVector
convertRingsListToVector(RingList ringList) // sink argument
{
    RingVector res;
    std::move(ringList.begin(), ringList.end(), std::back_inserter(res));
    return res;
}


// Polygon

PolygonAdapter::PolygonAdapter(const geolib3::Polygon2& polygon)
    : shell_(std::make_shared<geolib3::LinearRing2>(polygon.exteriorRing()))
{
    holes_.reserve(polygon.interiorRingsNumber());
    for (size_t hIdx = 0; hIdx < polygon.interiorRingsNumber(); ++hIdx) {
        holes_.emplace_back(
            std::make_shared<geolib3::LinearRing2>(polygon.interiorRingAt(hIdx)));
    }
}

void
PolygonAdapter::cacheArea() const
{
    double area = std::fabs(shell_.signedArea());
    for (const RingAdapter& hole : holes_) {
        area -= std::fabs(hole.signedArea());
    }
    area_ = area;
}

void
PolygonAdapter::cachePointsCount() const
{
    size_t count = shell_.pointsCount();
    for (const RingAdapter& hole : holes_) {
        count += hole.pointsCount();
    }
    pointsCount_ = count;
}

const RingAdapter&
PolygonAdapter::hole(size_t number) const
{
    if (number > holesCount()) {
        throw OutOfRangeException() << "Hole number out of range: " << number;
    }
    return holes_[number];
}

double
PolygonAdapter::area() const
{
    if (!area_) {
        cacheArea();
    }

    return *area_;
}

size_t
PolygonAdapter::pointsCount() const
{
    if (!pointsCount_) {
        cachePointsCount();
    }

    return *pointsCount_;
}

void
PolygonAdapter::normalize()
{
    shell_.setOrientation(Orientation::CCW);
    for (RingAdapter& hole : holes_) {
        hole.setOrientation(Orientation::CW);
    }
}

geolib3::Polygon2
PolygonAdapter::convertToGeolibPolygon() const
{
    auto shell = this->shell().ring();
    std::vector<geolib3::LinearRing2> holes;
    holes.reserve(holesCount());
    for (const RingAdapter& hole : holes_) {
        holes.push_back(hole.ring());
    }
    return geolib3::Polygon2(shell, holes, /*bool validate = */ false);
}

} // namespace geom_tools
} // namespace wiki
} // namespace maps
