#include <yandex/maps/wiki/geom_tools/polygonal/builder.h>

#include "../cut/polygonal_adapters.h"
#include "../utils.h"

#include <yandex/maps/wiki/geom_tools/conversion.h>

#include <maps/libs/geolib/include/conversion_geos.h>
#include <maps/libs/geolib/include/distance.h>
#include <maps/libs/geolib/include/static_geometry_searcher.h>

#include <geos/geom/LinearRing.h>

namespace maps {
namespace wiki {
namespace geom_tools {

namespace {

typedef uint64_t Id;

struct EdgeData {
    Id id;
    Id startNodeId;
    Id endNodeId;
    geolib3::PointsVector points;
};

struct RingData {
    std::map<Id, geolib3::Point2> nodes;
    geolib3::StaticGeometrySearcher<geolib3::Point2, Id> nodeSearcher;
    std::map<Id, EdgeData> edges;
    std::map<Id, Id> nodeToEdgeIds;
};

std::unique_ptr<RingData>
buildRingData(const geolib3::PolylinesVector& linestrings)
{
    std::unique_ptr<RingData> res(new RingData());
    Id genId = 0;
    for (const auto& ls : linestrings) {
        Id edgeId = ++genId;
        Id startNodeId = ++genId;
        Id endNodeId = ++genId;
        res->edges.insert({edgeId, {edgeId, startNodeId, endNodeId, ls.points()}});
        auto it = res->nodes.insert({startNodeId, ls.points().front()}).first;
        res->nodeSearcher.insert(&it->second, startNodeId);
        it = res->nodes.insert({endNodeId, ls.points().back()}).first;
        res->nodeSearcher.insert(&it->second, endNodeId);
        res->nodeToEdgeIds.insert({startNodeId, edgeId});
        res->nodeToEdgeIds.insert({endNodeId, edgeId});
    }
    res->nodeSearcher.build();
    return res;
}

GeolibLinearRingVector
geosGeomsToGeolibRings(const GeosGeometryPtrVector& geoms)
{
    GeolibLinearRingVector result;
    result.reserve(geoms.size());
    for (const auto& geom : geoms) {
        const geos::geom::LinearRing* ring =
            dynamic_cast<const geos::geom::LinearRing*>(geom);
        REQUIRE(ring, "Geom is not ring");
        result.push_back(geolib3::internal::geos2geolibGeometry(ring));
    }
    return result;
}

RingVector
geolibRingsToAdapters(const GeolibLinearRingVector& rings)
{
    RingVector result;
    result.reserve(rings.size());
    for (const auto& ring : rings) {
        result.push_back(RingAdapter(&ring));
    }
    return result;
}

GeosGeometryPtrVector
wikiPtrVectorToGeosPtrVector(const WikiGeomPtrVector& geoms)
{
    GeosGeometryPtrVector result;
    result.reserve(geoms.size());
    for (const auto& geom : geoms) {
        result.push_back(geom->geosGeometryPtr());
    }
    return result;
}

} // namespace


geolib3::LinearRing2 LinearRingBuilder::build(
    const WikiGeomPtrVector& linestrings,
    double tolerance)
{
    return LinearRingBuilder::build(
        wikiPtrVectorToGeosPtrVector(linestrings),
        tolerance);
}

geolib3::LinearRing2 LinearRingBuilder::build(
    const GeosGeometryPtrVector& linestrings,
    double tolerance)
{
    geolib3::PolylinesVector polylines;
    polylines.reserve(linestrings.size());
    for (const auto& ls : linestrings) {
        polylines.push_back(convert<GeosGeometry, geolib3::Polyline2>(*ls));
    }
    return LinearRingBuilder::build(polylines, tolerance);
}

geolib3::LinearRing2 LinearRingBuilder::build(
    const geolib3::PolylinesVector& linestrings,
    double tolerance)
{
    REQUIRE(!linestrings.empty(), "Empty linestrings in ring");

    if (linestrings.size() == 1) {
        const auto& points = linestrings.front().points();
        if (points.size() < 4) {
            throw RingBuildError() << " too few points in ring: " << points.size();
        }
        if (geolib3::distance(points.front(), points.back()) > tolerance) {
            throw RingBuildError() << " ring must be formed from closed linestring";
        }
        return geolib3::LinearRing2(points);
    }

    auto ringData = buildRingData(linestrings);
    const auto& startEdge = ringData->edges.begin()->second;
    geolib3::PointsVector points = startEdge.points;
    std::set<Id> visitedNodeIds = {startEdge.startNodeId, startEdge.endNodeId};
    auto startNodeId = startEdge.startNodeId;

    auto findNextNode = [&] (Id nodeId)
    {
        auto nodeSearchRes = ringData->nodeSearcher.find(
            geolib3::resizeByValue(ringData->nodes.at(nodeId).boundingBox(), tolerance));
        boost::optional<Id> nextEdgeNodeId;
        for (auto it = nodeSearchRes.first; it != nodeSearchRes.second; ++it) {
            if (it->value() == nodeId) {
                continue;
            }
            if (nextEdgeNodeId) {
                throw RingBuildError()
                    << " more than two edges are connected to one node in ring"
                    << " near point (" << it->geometry().x() << ", " << it->geometry().y() << ")";
            }
            nextEdgeNodeId = it->value();
        }
        if (!nextEdgeNodeId) {
            const auto& nodePos = ringData->nodes.at(nodeId);
            throw RingBuildError() << " dangling node in ring"
                << " near point (" << nodePos.x() << ", " << nodePos.y() << ")";
        }
        return *nextEdgeNodeId;
    };

    Id curNodeId = findNextNode(startEdge.endNodeId);
    ringData->edges.erase(ringData->edges.begin());
    while (!ringData->edges.empty() && !visitedNodeIds.count(curNodeId)) {
        auto edgeId = ringData->nodeToEdgeIds.at(curNodeId);
        auto edgeIt = ringData->edges.find(edgeId);
        REQUIRE(edgeIt != ringData->edges.end(), "Edge met twice");
        const auto& edge = edgeIt->second;
        if (curNodeId == edge.startNodeId) {
            points.insert(points.end(), std::next(edge.points.begin()), edge.points.end());
            curNodeId = edge.endNodeId;
        } else {
            points.insert(points.end(), std::next(edge.points.rbegin()), edge.points.rend());
            curNodeId = edge.startNodeId;
        }
        visitedNodeIds.insert(edge.startNodeId);
        visitedNodeIds.insert(edge.endNodeId);
        ringData->edges.erase(edgeIt);
        curNodeId = findNextNode(curNodeId);
    }

    if (!ringData->edges.empty()) {
        throw RingBuildError() << " ring consists of multiple components";
    }
    if (curNodeId != startNodeId) {
        throw RingBuildError() << " ring is not closed";
    }

    return geolib3::LinearRing2(points);
}


GeolibPolygonVector
PolygonBuilder::build(
    const WikiGeomPtrVector& shells, const WikiGeomPtrVector& holes,
    ValidateResult validateResult)
{
    return PolygonBuilder::build(
        wikiPtrVectorToGeosPtrVector(shells),
        wikiPtrVectorToGeosPtrVector(holes),
        validateResult);
}

GeolibPolygonVector
PolygonBuilder::build(
    const GeosGeometryPtrVector& shells, const GeosGeometryPtrVector& holes,
    ValidateResult validateResult)
{
    return PolygonBuilder::build(
        geosGeomsToGeolibRings(shells),
        geosGeomsToGeolibRings(holes),
        validateResult);
}

GeolibPolygonVector
PolygonBuilder::build(
    const GeolibLinearRingVector& shells, const GeolibLinearRingVector& holes,
    ValidateResult validateResult)
{
    return buildPolygons(
        geolibRingsToAdapters(shells),
        geolibRingsToAdapters(holes),
        validateResult);
}

} // namespace geom_tools
} // namespace wiki
} // namespace maps
