#include <yandex/maps/wiki/graph/strongly_connected_components.h>

#include <maps/libs/common/include/exception.h>
#include <stack>

namespace maps::wiki::graph {

struct StronglyConnectedComponentsFinder::NodeInfo
{
    NodeInfo()
        : nodeIndex(0)
        , parentNodeId(0)
        , sccRootIndex(0)
    {}

    NodeInfo(
            size_t nodeIndex,
            size_t parentNodeId,
            size_t sccRootIndex)
        : nodeIndex(nodeIndex)
        , parentNodeId(parentNodeId)
        , sccRootIndex(sccRootIndex)
    {}

    size_t nodeIndex;
    size_t parentNodeId;
    size_t sccRootIndex;
};

StronglyConnectedComponentsFinder::StronglyConnectedComponentsFinder() = default;

StronglyConnectedComponentsFinder::~StronglyConnectedComponentsFinder() = default;

size_t
StronglyConnectedComponentsFinder::componentIndex(NodeID nodeId) const
{
    auto iter = nodeIdToComponent_.find(nodeId);
    ASSERT(iter != nodeIdToComponent_.end());
    return iter->second;
}

void
StronglyConnectedComponentsFinder::exploreNode(NodeID fromNodeId, GetOutEdges outEdges)
{
    // Already visited in one of previous calls to exploreNode()
    if (nodeIdToNodeInfo_.count(fromNodeId)) {
        return;
    }

    std::stack<NodeID> dfsStack;
    std::stack<NodeID> sccStack;
    size_t newNodeIndex = 1;

    dfsStack.push(fromNodeId);

    while (!dfsStack.empty()) {
        auto currentNodeId = dfsStack.top();
        auto& currentNodeInfo = nodeIdToNodeInfo_[currentNodeId];

        if (currentNodeInfo.nodeIndex) {
            auto& parentNodeInfo = nodeIdToNodeInfo_[currentNodeInfo.parentNodeId];
            parentNodeInfo.sccRootIndex = std::min(
                parentNodeInfo.sccRootIndex, currentNodeInfo.sccRootIndex);

            if (currentNodeInfo.nodeIndex == currentNodeInfo.sccRootIndex) {
                components_.push_back({});

                while (!sccStack.empty()) {
                    auto nodeId = sccStack.top();
                    sccStack.pop();

                    components_.back().push_back(nodeId);
                    nodeIdToComponent_[nodeId] = components_.size() - 1;
                    nodeIdToNodeInfo_[nodeId].sccRootIndex = 0;

                    if (nodeId == currentNodeId) {
                        break;
                    }
                }
            }
            dfsStack.pop();
        } else {
            currentNodeInfo.nodeIndex = newNodeIndex;
            currentNodeInfo.sccRootIndex = newNodeIndex;
            ++newNodeIndex;
            sccStack.push(currentNodeId);

            for (const auto& edge : outEdges(currentNodeId)) {
                auto endNodeId = edge.endNodeId();
                auto& endNodeInfo = nodeIdToNodeInfo_[endNodeId];

                if (!endNodeInfo.nodeIndex) {
                    endNodeInfo.parentNodeId = currentNodeId;
                    dfsStack.push(endNodeId);
                } else if (endNodeInfo.sccRootIndex) {
                    currentNodeInfo.sccRootIndex = std::min(
                        currentNodeInfo.sccRootIndex, endNodeInfo.nodeIndex);
                }
            }
        }
    }
}

} // namespace maps::wiki::graph
