#include <yandex/maps/wiki/groupedit/actions/delete.h>
#include <yandex/maps/wiki/groupedit/object.h>
#include <yandex/maps/wiki/groupedit/relation.h>
#include <yandex/maps/wiki/groupedit/session.h>

#include <maps/libs/common/include/exception.h>

#include <algorithm>
#include <string>
#include <unordered_map>
#include <unordered_set>

namespace maps {
namespace wiki {
namespace groupedit {
namespace actions {

namespace {

typedef std::unordered_set<TObjectId> TObjectIdSet;
typedef std::unordered_map<TObjectId, TObjectId> TObjectIdMap;

const std::string START_ROLE = "start";
const std::string END_ROLE = "end";
const std::string PART_ROLE = "part";
const std::string LN_PART_ROLE = "ln_part";
const std::string FC_PART_ROLE = "fc_part";
const std::string CENTER_ROLE = "center";
const std::string ASSIGNED_ROLE = "assigned";
const std::string ASSIGNED_THREAD_STOP_ROLE = "assigned_thread_stop";
const std::string ASSIGNED_THREAD_ROLE = "assigned_thread";
const std::string PREVIOUS_ROLE = "previous";
const std::string ASSOCIATED_WITH_ROLE = "associated_with";
const std::string ADDR_ASSOCIATED_WITH_ROLE = "addr_associated_with";
const std::string VIA_ROLE = "via";

const std::string MODEL3D_CATEGORY = "model3d";
const std::string COND_CATEGORY_PREFIX = "cond";
const std::string COND_DT_CATEGORY = "cond_dt";

const std::string TRANSPORT_STOP_CATEGORY = "transport_stop";
const std::string TRANSPORT_WATERWAY_STOP_CATEGORY = "transport_waterway_stop";
const std::string TRANSPORT_THREAD_STOP_CATEGORY = "transport_thread_stop";
const std::string FREQ_DT_CATEGORY = "freq_dt";

const std::string GROUP_DELETED_ACTION = "group-deleted";



bool isTransportStop(const std::string& category)
{
    return category == TRANSPORT_STOP_CATEGORY
        || category == TRANSPORT_WATERWAY_STOP_CATEGORY;
}

bool isJunctionCategory(const std::string& category)
{
    return category.ends_with("_jc");
}

bool isElementCategory(const std::string& category)
{
    return category.ends_with("_el");
}

bool isCenterCategory(const std::string& category)
{
    return category.ends_with("_cnt");
}

bool isFaceElement(const Object& obj)
{
    if (!isElementCategory(obj.category())) {
        return false;
    }

    for (const auto& rel : obj.relations()) {
        if (rel.type() == Relation::Type::Master
                && rel.otherCategory().ends_with("_fc")) {
            return true;
        }
    }

    return false;
}

bool isLinearElement(const Object& obj)
{
    if (!isElementCategory(obj.category())) {
        return false;
    }

    return !isFaceElement(obj);
}

bool isMasterOfPartRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave
        && (rel.role() == PART_ROLE
            || rel.role() == LN_PART_ROLE
            || rel.role() == FC_PART_ROLE);
}

bool isSlaveOfPartRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Master
        && (rel.role() == PART_ROLE
            || rel.role() == LN_PART_ROLE
            || rel.role() == FC_PART_ROLE);
}

bool isMasterOfNameRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave
        && rel.otherCategory().ends_with("_nm");
}

bool isMasterOfCenterRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave && rel.role() == CENTER_ROLE;
}

bool isSlaveOfCenterRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Master && rel.role() == CENTER_ROLE;
}

bool isMasterOfAssociatedRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave
        && (rel.role() == ASSOCIATED_WITH_ROLE
            || rel.role() == ADDR_ASSOCIATED_WITH_ROLE);
};

bool isMasterOfAssignedRel(const Relation& rel)
{
    return rel.role() == ASSIGNED_ROLE && rel.type() == Relation::Type::Slave;
}

bool isSlaveOfAssignedRel(const Relation& rel)
{
    return rel.role() == ASSIGNED_ROLE && rel.type() == Relation::Type::Master;
}

bool isMaterOfPreviousRel(const Relation& rel)
{
    return rel.role() == PREVIOUS_ROLE && rel.type() == Relation::Type::Slave;
}

bool isAssignedThreadStopRel(const Relation& rel)
{
    return rel.role() == ASSIGNED_THREAD_STOP_ROLE;
}

bool isAssignedThreadRel(const Relation& rel)
{
    return rel.role() == ASSIGNED_THREAD_ROLE;
}

bool isMasterOfJcElRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave
        && (rel.role() == START_ROLE || rel.role() == END_ROLE);
}

bool isSlaveOfJcElRel(const Relation& rel)
{
    return rel.type() == Relation::Type::Master
        && (rel.role() == START_ROLE || rel.role() == END_ROLE);
}

bool isSlaveOfCond(const Relation& rel)
{
    return rel.type() == Relation::Type::Master
        && rel.otherCategory().starts_with(COND_CATEGORY_PREFIX);
}

bool isRelToCondDt(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave
        && rel.otherCategory() == COND_DT_CATEGORY;
}

bool isRelToFreqDt(const Relation& rel)
{
    return rel.type() == Relation::Type::Slave
        && rel.otherCategory() == FREQ_DT_CATEGORY;
}

bool isRelToModel3d(const Relation& rel)
{
    return rel.type() == Relation::Type::Master
        && rel.otherCategory() == MODEL3D_CATEGORY;
}

template<class Container>
bool areAllRelationsTo(
        const Object& obj,
        const std::function<bool(const Relation&)>& relPredicate,
        const Container& related)
{
    for (const auto& rel : obj.relations()) {
        if (relPredicate(rel) && !related.count(rel.otherId())) {
            return false;
        }
    }
    return true;
}

void collectByRelations(
        const Object& obj,
        const std::function<bool(const Relation&)>& relPredicate,
        TObjectIdSet& output)
{
    for (const auto& rel : obj.relations()) {
        if (relPredicate(rel)) {
            output.insert(rel.otherId());
        }
    }
}

void collectNamedObject(
        const Object& obj,
        TObjectIdSet& output)
{
    output.insert(obj.id());
    collectByRelations(obj, isMasterOfNameRel, output);
}

struct DeleteContext
{
    TObjectIdSet conditionsToDelete;

    TObjectIdSet candidateJunctions;
    TObjectIdSet candidateLinearObjects;
    TObjectIdSet candidateFaceElements;
    TObjectIdSet candidateFaces;
    TObjectIdSet candidateContourObjects;

    TObjectIdSet threadStopsToDelete;
    TObjectIdSet candidateTransportRoutes;
};

void deleteEmptyTransportRoute(
        const Session& session,
        DeleteContext& context,
        TObjectIdSet& toDelete)
{
    session.query(context.candidateTransportRoutes).visit(
        [&](const Object& route) {
            const bool empty = areAllRelationsTo(route, isMasterOfAssignedRel, toDelete)
                && areAllRelationsTo(route, isAssignedThreadRel, toDelete);
            if (empty) {
                collectNamedObject(route, toDelete);
            }
        }
    );
}

struct ThreadStop {
    TObjectId id;
    TObjectId previousId;

    ThreadStop(TObjectId id, TObjectId previousId)
        : id(id)
        , previousId(previousId)
    {}
};

typedef std::vector<ThreadStop> ThreadStops;
typedef std::unordered_map<TObjectId, ThreadStops> ThreadIdToThreadStops;

void orderThreadStops(ThreadStops& threadStops)
{
    TObjectId previousId = 0;
    for (auto to = threadStops.begin(); to != threadStops.end(); ++to) {
        auto from = std::find_if(
            to, threadStops.end(),
            [&previousId](const ThreadStop& threadStop) {
                return threadStop.previousId == previousId;
            }
        );

        REQUIRE(from != threadStops.end(), "Missed thread_stop " << previousId);

        std::iter_swap(from, to);
        previousId = to->id;
    }
}

TObjectIdSet collectThreads(const Session& session, const TObjectIdSet& threadStops)
{
    TObjectIdSet threads;
    session.query(threadStops).visit(
        [&threads](const Object& threadStop) {
            collectByRelations(threadStop, isSlaveOfPartRel, threads);
        }
    );

    return threads;
}

ThreadIdToThreadStops constructThreads(const Session& session, const TObjectIdSet& threadStops)
{
    ThreadIdToThreadStops threadIdToThreadStops;

    session.query(threadStops).visit(
        [&](const Object& threadStop) {
            TObjectId threadId = 0;
            TObjectId previousId = 0;

            for (const auto& rel: threadStop.relations()) {
                if (isMaterOfPreviousRel(rel)) {
                    previousId = rel.otherId();
                } else if (isSlaveOfPartRel(rel)) {
                    threadId = rel.otherId();
                }

                if (threadId && previousId) {
                    break;
                }
            }

            REQUIRE(
                threadId,
                "Missed relation with role 'part', thread_stop " << threadStop.id()
            );

            threadIdToThreadStops[threadId].emplace_back(
                threadStop.id(),
                previousId
            );
        }
    );


    return threadIdToThreadStops;
}

void dropThreadStops(
    ThreadIdToThreadStops& threadIdToThreadStops,
    const TObjectIdSet& threadStopsToDelete)
{
    for (auto& pair: threadIdToThreadStops) {
        ThreadStops& threadStops = pair.second;

        orderThreadStops(threadStops);
        const auto end = std::remove_if(
            threadStops.begin(),
            threadStops.end(),
            [&](ThreadStop& threadStop) {
                return threadStopsToDelete.count(threadStop.id);
            }
        );

        threadStops.erase(end, threadStops.end());
    }
}

std::vector<TCommitId> recontructThreadOrDeleteIfEmpty(
        const Session& session,
        TUserId author,
        DeleteContext& context,
        TObjectIdSet& toDelete)
{
    const TObjectIdSet threads = collectThreads(session, context.threadStopsToDelete);

    TObjectIdSet threadStops;
    session.query(threads).visit(
        [&](const Object& thread) {
            if (areAllRelationsTo(thread, isMasterOfPartRel, toDelete)) {
                toDelete.insert(thread.id());
                collectByRelations(thread, isRelToFreqDt, toDelete);
            } else {
                collectByRelations(thread, isMasterOfPartRel, threadStops);
            }
        }
    );

    ThreadIdToThreadStops threadIdToThreadStops = constructThreads(session, threadStops);
    dropThreadStops(threadIdToThreadStops, context.threadStopsToDelete);

    std::vector<TObjectId> threadStopsToUpdatePreviousRelation;
    TObjectIdMap threadStopIdToNewPreviousId;
    for (const auto& pair: threadIdToThreadStops) {
        const auto& threadStops = pair.second;

        TObjectId previousId = 0;
        for (const auto threadStop: threadStops) {
            if (threadStop.previousId != previousId) {
                threadStopsToUpdatePreviousRelation.push_back(threadStop.id);
                threadStopIdToNewPreviousId[threadStop.id] = previousId;
            }
            previousId = threadStop.id;
        }
    }

    return session.query(threadStopsToUpdatePreviousRelation).update(
        GROUP_DELETED_ACTION,
        author,
        [&](Object& threadStop) {
            const TObjectId previousId = threadStopIdToNewPreviousId.at(threadStop.id());
            if (previousId) {
                threadStop.addRelationToSlave(
                    previousId,
                    TRANSPORT_THREAD_STOP_CATEGORY,
                    PREVIOUS_ROLE
                );
            }
        }
    );
}

void deleteEmptyLinearObject(
        const Session& session,
        DeleteContext& context,
        TObjectIdSet& toDelete)
{
    session.query(context.candidateLinearObjects).visit(
        [&](const Object& obj) {
            if (areAllRelationsTo(obj, isMasterOfPartRel, toDelete)
                  && areAllRelationsTo(obj, isMasterOfAssociatedRel, toDelete)) {
                collectNamedObject(obj, toDelete);
            }
        }
    );
}

void deleteJunctionIfNoRelatedElement(
        const Session& session,
        DeleteContext& context,
        TObjectIdSet& toDelete)
{
    session.query(context.candidateJunctions).visit(
        [&](const Object& junction) {
            if (areAllRelationsTo(junction, isSlaveOfJcElRel, toDelete)) {
                toDelete.insert(junction.id());
                collectByRelations(junction, isSlaveOfCond, context.conditionsToDelete);
            }
        }
    );
}

void deleteCondition(
        const Session& session,
        DeleteContext& context,
        TObjectIdSet& toDelete)
{
    session.query(context.conditionsToDelete).visit(
        [&](const Object& condition) {
            toDelete.insert(condition.id());
            collectByRelations(condition, isRelToCondDt, toDelete);
        }
    );
}

void deleteElementWithRelatedFaceAndCountur(
    const Session& session,
    DeleteContext& context,
    TObjectIdSet& toDelete)
{
    TObjectIdSet facesWithin;
    session.query(context.candidateFaces).visit(
        [&](const Object& face) {
            if (areAllRelationsTo(face, isMasterOfPartRel, context.candidateFaceElements)) {
                facesWithin.insert(face.id());
                collectByRelations(face, isSlaveOfPartRel, context.candidateContourObjects);
            }
        }
    );

    TObjectIdSet deletedFaces;
    session.query(context.candidateContourObjects).visit(
        [&](const Object& obj) {
            if (!areAllRelationsTo(obj, isMasterOfPartRel, facesWithin)) {
                return;
            }

            collectByRelations(obj, isMasterOfCenterRel, toDelete);
            collectByRelations(obj, isMasterOfPartRel, toDelete);
            collectByRelations(obj, isMasterOfPartRel, deletedFaces);

            if (areAllRelationsTo(obj, isMasterOfAssociatedRel, toDelete)) {
                collectNamedObject(obj, toDelete);
            }
        }
    );

    session.query(context.candidateFaceElements).visit(
        [&](const Object& element) {
            if (areAllRelationsTo(element, isSlaveOfPartRel, deletedFaces)) {
                toDelete.insert(element.id());
            }
        }
    );
}

} // namespace

std::vector<TCommitId> deleteObjects(
        const Session& session,
        const std::vector<RevGeomFilter>& revGeomFilters,
        TUserId author)
{
    TObjectIdSet toDelete;

    DeleteContext context;

    for (const auto& revGeomFilter : revGeomFilters) {
        session.query(revGeomFilter).visit(
            [&](const Object& obj) {
                if (isLinearElement(obj)) {
                    collectByRelations(obj, isSlaveOfPartRel, context.candidateLinearObjects);
                    collectByRelations(obj, isMasterOfJcElRel, context.candidateJunctions);
                    collectByRelations(obj, isSlaveOfCond, context.conditionsToDelete);
                    toDelete.insert(obj.id());
                } else if (isFaceElement(obj)) {
                    context.candidateFaceElements.insert(obj.id());
                    collectByRelations(obj, isSlaveOfPartRel, context.candidateFaces);
                    collectByRelations(obj, isMasterOfJcElRel, context.candidateJunctions);
                } else if (isTransportStop(obj.category())) {
                    collectNamedObject(obj, toDelete);
                    collectByRelations(obj, isSlaveOfAssignedRel, context.candidateTransportRoutes);
                    collectByRelations(obj, isAssignedThreadStopRel, context.threadStopsToDelete);
                    collectByRelations(obj, isAssignedThreadStopRel, toDelete);
                } else if (isJunctionCategory(obj.category())) {
                    context.candidateJunctions.insert(obj.id());
                } else if (isCenterCategory(obj.category())) {
                    collectByRelations(obj, isSlaveOfCenterRel, context.candidateContourObjects);
                } else {
                    collectNamedObject(obj, toDelete);
                    collectByRelations(obj, isRelToModel3d, toDelete);
                }
        });
    }

    deleteEmptyLinearObject(session, context, toDelete);
    deleteElementWithRelatedFaceAndCountur(session, context, toDelete);
    deleteJunctionIfNoRelatedElement(session, context, toDelete);
    deleteCondition(session, context, toDelete);
    auto commitIds = recontructThreadOrDeleteIfEmpty(session, author, context, toDelete);
    deleteEmptyTransportRoute(session, context, toDelete);

    const auto deleteCommitIds = session.query(toDelete).update(
        GROUP_DELETED_ACTION,
        author,
        [&](Object& obj) {
            obj.setDeleted();
        }
    );
    commitIds.insert(commitIds.end(), deleteCommitIds.begin(), deleteCommitIds.end());

    return commitIds;
}


} // namespace actions
} // namespace groupedit
} // namespace wiki
} // namespace maps
