#include "commit_writer.h"

#include <yandex/maps/wiki/revision/revisionsgateway.h>

#include <maps/libs/common/include/exception.h>

namespace maps {
namespace wiki {
namespace groupedit {

namespace {

const std::string ACTION_ATTR = "action";
const std::string SOURCE_ATTR = "source";
const std::string LONG_TASK_STR = "long-task";

} // namespace

void CommitWriter::add(ObjectDiff objectDiff)
{ diffQueue_.push(std::move(objectDiff)); }

std::vector<TCommitId> CommitWriter::write(
        revision::RevisionsGateway& gateway,
        TUserId author,
        std::string commitAction,
        size_t maxCommitSize)
{
    REQUIRE(maxCommitSize, "Invalid commit size");

    diffQueue_.finish();

    std::list<ObjectDiff> buffer;
    ObjectDiff::NewRevisionsData currentRevisions;
    std::vector<TCommitId> commitIds;

    Attributes attributes = {
        {ACTION_ATTR, std::move(commitAction)},
        {SOURCE_ATTR, LONG_TASK_STR}};

    auto createCommit = [&](ObjectDiff::NewRevisionsData& revisions)
    {
        for (auto& newRev : revisions) {
            if (newRev.first == revision::RevisionID()) {
                newRev.first = gateway.acquireObjectId();
            }
        }

        commitIds.push_back(gateway.createCommit(
            revisions, author, attributes).id());
    };

    do {
        diffQueue_.popAll(buffer);

        for (auto& diff : buffer) {
            if (diff.revisionsCount() == 0) {
                continue;
            }

            if (!currentRevisions.empty() &&
                    currentRevisions.size() + diff.revisionsCount()
                        > maxCommitSize) {
                createCommit(currentRevisions);
                currentRevisions.clear();
            }
            currentRevisions.splice(
                std::end(currentRevisions),
                ObjectDiff::extractCommitData(std::move(diff)));
        }
    } while (!buffer.empty());

    if (!currentRevisions.empty()) {
        createCommit(currentRevisions);
    }

    return commitIds;
}

} // namespace groupedit
} // namespace wiki
} // namespace maps
