#include "geom.h"

#include <maps/libs/common/include/exception.h>
#include <maps/libs/geolib/include/intersection.h>
#include <maps/libs/geolib/include/spatial_relation.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/polygon.h>
#include <maps/libs/geolib/include/polyline.h>
#include <maps/libs/geolib/include/variant.h>

#include <cstdint>

namespace maps {
namespace wiki {
namespace groupedit {
namespace utils {

namespace {

bool spatialRelation(
        const geolib3::Polygon2& polygon,
        const std::string& wkb,
        geolib3::SpatialRelation relationType)
{
    auto geometryVariant = geolib3::WKB::read<geolib3::SimpleGeometryVariant>(wkb);
    return geometryVariant.visit(
        [&](const auto& geometry) {
            return geolib3::spatialRelation(polygon, geometry, relationType);
        });
}

} // namespace

bool contains(const geolib3::Polygon2& polygon, const std::string& wkb)
{ return spatialRelation(polygon, wkb, geolib3::Contains); }

bool intersects(const geolib3::Polygon2& polygon, const std::string& wkb)
{ return spatialRelation(polygon, wkb, geolib3::Intersects); }

bool intersects(const geolib3::Polyline2& polylineLeft, const std::string& wkb)
{
    struct IntersectsVisitor
    {
        const geolib3::Polyline2& polylineLeft_;

        explicit IntersectsVisitor(const geolib3::Polyline2& polylineLeft)
            : polylineLeft_(polylineLeft)
        {}

        bool operator()(const geolib3::Point2& point) const
        {
            return !!polylineLeft_.segmentIndex(point);
        }

        bool operator()(const geolib3::Polyline2& polyline) const
        {
            return geolib3::intersects(polyline, polylineLeft_);
        }

        bool operator()(const geolib3::Polygon2& polygon) const
        {
            return geolib3::spatialRelation(polygon, polylineLeft_, geolib3::Intersects);
        }
    };

    auto geometryVariant = geolib3::WKB::read<geolib3::SimpleGeometryVariant>(wkb);
    return geometryVariant.visit(IntersectsVisitor(polylineLeft));
}

} // namespace utils
} // namespace groupedit
} // namespace wiki
} // namespace maps
