#include <yandex/maps/wiki/http/fbapi/gateway.h>
#include <yandex/maps/wiki/http/fbapi/exception.h>

#include "../http_utils.h"

#include <maps/libs/auth/include/tvm.h>
#include <maps/libs/http/include/url.h>
#include <maps/libs/json/include/builder.h>

#include <map>
#include <sstream>

namespace maps::wiki::fbapi {

namespace {

http::HeaderMap makeHeaderMap(const Configuration& configuration)
{
    http::HeaderMap headerMap;
    if (configuration.tvmTicketProvider()) {
        headerMap.emplace(auth::SERVICE_TICKET_HEADER, configuration.tvmTicketProvider()());
    }
    return headerMap;
}

} // unnamed namespace

Gateway::Gateway(
    Configuration configuration,
    const maps::common::RetryPolicy& retryPolicy)
        : configuration_(std::move(configuration))
        , retryPolicy_(retryPolicy)
{}

Task
Gateway::taskById(const std::string& taskId, WithHistory withHistory) const
{
    http::URL url(configuration_.baseUrl() + "/v1/tasks/" + taskId);
    if (withHistory == WithHistory::Yes) {
        url.addParam("with_history", "true");
    }

    http::Client httpClient;
    auto [response, status] = httpClient.get(
        url,
        makeHeaderMap(configuration_),
        retryPolicy_
    );

    if (status == http_status::OK) {
        return Task(json::Value::fromString(response));
    }
    if (status == http_status::NOT_FOUND) {
        throw KeyNotFoundError();
    }
    throw FeedbackError() << "Unexpected http code: " << status
        << "; GET url: " << url;
}

Tasks
Gateway::tasksByFilter(
    const TasksFilter& tasksFilter,
    const PageParams& pageParams,
    WithHistory withHistory) const
{
    http::URL url(configuration_.baseUrl() + "/v1/tasks");
    url.addParam("status", toString(tasksFilter.status()));
    url.addParam("service", toString(tasksFilter.service()));
    if (pageParams.offset()) {
        url.addParam("offset", std::to_string(pageParams.offset().value()));
    }
    if (withHistory == WithHistory::Yes) {
        url.addParam("with_history", "true");
    }

    http::Client httpClient;
    auto [response, status] = httpClient.get(
        url,
        makeHeaderMap(configuration_),
        retryPolicy_
    );

    if (status == http_status::OK) {
        return Tasks(json::Value::fromString(response));
    }
    throw FeedbackError() << "Unexpected http code: " << status
        << "; GET url: " << url;
}

void
Gateway::changeTaskById(const std::string& taskId, const ChangeTaskParams& params)
{
    if (!params.newService() && !params.newStatus()) {
        throw FeedbackError() << "Either new service or status must be specified";
    }

    http::URL url(configuration_.baseUrl() + "/v1/tasks/" + taskId);

    auto body = (json::Builder{} << params).str();

    http::Client httpClient;
    auto singlePerform = [&]() {
        http::Request req(httpClient, http::PUT, url);
        for (const auto& pair: makeHeaderMap(configuration_)) {
            req.addHeader(pair);
        }
        req.addHeader("Content-Type", "application/json");
        req.setContent(body);
        return req.perform();
    };

    auto response = performHttpRequest(singlePerform, retryPolicy_);

    if (response.status() == http_status::NOT_FOUND) {
        throw KeyNotFoundError();
    }
    if (response.status() == http_status::BAD_REQUEST) {
        throw BadRequestError(response.readBody());
    }
    if (response.status() != http_status::OK) {
        throw FeedbackError() << "Unexpected http code: " << response.status()
            << "; PUT url: " << url;
    }
}

void visitTasks(
    const IGateway& gateway,
    const TasksFilter& tasksFilter,
    WithHistory withHistory,
    std::function<bool(const Task&)> visitor)
{
    PageParams pageParams;
    pageParams.setOffset(0);

    while (true) {
        const auto tasks = gateway.tasksByFilter(tasksFilter, pageParams, withHistory).tasks();
        if (tasks.empty()) {
            break;
        }
        for (const auto& task : tasks) {
            auto needContinue = visitor(task);
            if (!needContinue) {
                return;
            }
        }
        pageParams.setOffset(pageParams.offset().value() + tasks.size());
    }
}

} // namespace maps::wiki::fbapi
