#pragma once

#include <yandex/maps/wiki/common/string_utils.h>
#include <yandex/maps/wiki/http/fbapi/models/task.h>
#include <yandex/maps/wiki/http/fbapi/models/common.h>
#include <yandex/maps/wiki/http/fbapi/models/version.h>
#include <yandex/maps/wiki/http/fbapi/exception.h>

#include <maps/libs/json/include/value.h>
#include <maps/libs/geolib/include/point.h>
#include <maps/libs/geolib/include/conversion.h>
#include <util/system/compiler.h>

#include <optional>
#include <type_traits>
#include <vector>

namespace maps::wiki::fbapi {

#define PARSING_REQUIRE(cond, msg) \
    if (Y_LIKELY(cond)) {} else \
        throw FeedbackError() << "JSON parsing error: " << msg;

std::string
toShortString(const json::Value& value);


template<typename T> struct is_vector : public std::false_type {};

template<typename T, typename A>
struct is_vector<std::vector<T, A>> : public std::true_type {};


template<typename T> struct is_object : public std::false_type {};

template<> struct is_object<AnswerContext> : public std::true_type {};
template<> struct is_object<FormContext> : public std::true_type {};
template<> struct is_object<Gate> : public std::true_type {};
template<> struct is_object<OriginalTaskMetadata> : public std::true_type {};
template<> struct is_object<OriginalTask> : public std::true_type {};
template<> struct is_object<QuestionContext> : public std::true_type {};
template<> struct is_object<RouteQuestionSegment> : public std::true_type {};
template<> struct is_object<RouteError> : public std::true_type {};
template<> struct is_object<RouteAnswerSegment> : public std::true_type {};
template<> struct is_object<RouteErrorProhibitingSignContext> : public std::true_type {};
template<> struct is_object<TaskChange> : public std::true_type {};
template<> struct is_object<Task> : public std::true_type {};
template<> struct is_object<Tasks> : public std::true_type {};
template<> struct is_object<ToponymAddressComponent> : public std::true_type {};
template<> struct is_object<Toponym> : public std::true_type {};
template<> struct is_object<VehicleRestrictions> : public std::true_type {};


template<class T>
typename std::enable_if<std::is_same_v<std::string, T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isString(),
        "expected string value '" << toShortString(value) << "'"
    )
    return value.as<T>();
}

template<class T>
typename std::enable_if<std::is_arithmetic_v<T> && !std::is_same_v<bool, T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isNumber(),
        "expected number value '" << toShortString(value) << "'"
    )
    return value.as<T>();
}

template<class T>
typename std::enable_if<std::is_same_v<bool, T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isBool(),
        "expected bool value '" << toShortString(value) << "'"
    )
    return value.as<T>();
}

template<class T>
typename std::enable_if<std::is_enum_v<T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isString(),
        "expected string value '" << toShortString(value) << "'"
    )
    return enum_io::fromString<T>(value.as<std::string>());
}

template<class T>
typename std::enable_if<std::is_same_v<chrono::TimePoint, T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isString(),
        "expected string value '" << toShortString(value) << "'"
    )
    return chrono::parseIsoDateTime(value.as<std::string>());
}

template<class T>
typename std::enable_if<std::is_same_v<geolib3::Point2, T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isObject(),
        "expected object value '" << toShortString(value) << "'"
    )
    PARSING_REQUIRE(
        value.hasField("lon"),
        "field 'lon' expected in value '" << toShortString(value) << "'"
    )
    PARSING_REQUIRE(
        value.hasField("lat"),
        "field 'lat' expected in value '" << toShortString(value) << "'"
    )

    T geoPoint{
        value["lon"].as<double>(),
        value["lat"].as<double>()
    };

    return geolib3::convertGeodeticToMercator(geoPoint);
}

template<class T>
typename std::enable_if<std::is_same_v<Version, T>, T>::type
fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isString(),
        "expected string value '" << toShortString(value) << "'"
    );

    auto versionStr = value.as<std::string>();

    auto strParts = common::split(versionStr, ".");

    PARSING_REQUIRE(
        strParts.size() >= 1 && strParts.size() <= 3,
        "Invalid number of parts in version '" << versionStr << "'"
    );

    std::vector<unsigned> intParts;

    try {
        std::transform(
            strParts.begin(), strParts.end(),
            std::back_inserter(intParts),
            [](const auto& str){ return std::stoul(str); }
        );
    } catch (const std::logic_error&) {
        PARSING_REQUIRE(
            false,
            "Cannot convert version '" << versionStr << "' to set of numbers"
        );
    }

    std::array<unsigned, 3> args = {0, 0, 0};
    std::copy(intParts.begin(), intParts.end(), args.begin());
    return Version(args[0], args[1], args[2]);
}

template<class T, typename std::enable_if<is_object<T>::value, bool>::type = true>
T fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isObject(),
        "expected object value '" << toShortString(value) << "'"
    )
    return T(value);
}

template<class T, typename std::enable_if<is_vector<T>::value, bool>::type = true>
T fromJson(const json::Value& value)
{
    PARSING_REQUIRE(
        value.isArray(),
        "expected array value '" << toShortString(value) << "'"
    )
    T result;
    for (auto it = value.begin(); it != value.end(); ++it) {
        result.push_back(fromJson<typename T::value_type>(*it));
    }
    return result;
}


template<class T, typename std::enable_if<!is_vector<T>::value, bool>::type = true>
T parseField(const json::Value& value, const std::string& fieldName)
{
    PARSING_REQUIRE(
        value.hasField(fieldName),
        "field '" << fieldName << "' expected in value '" << toShortString(value) << "'"
    )
    return fromJson<T>(value[fieldName]);
}

template<class T, typename std::enable_if<is_vector<T>::value, bool>::type = true>
T parseField(const json::Value& value, const std::string& fieldName)
{
    if (!value.hasField(fieldName)) {
        return {};
    }
    return fromJson<T>(value[fieldName]);
}

template<class T>
std::optional<T> parseOptField(const json::Value& value, const std::string& fieldName)
{
    if (!value.hasField(fieldName)) {
        return std::nullopt;
    }
    return fromJson<T>(value[fieldName]);
}

} // namespace maps::wiki::fbapi
