#pragma once

/**
 * Blackbox HTTP API client implementation
 * https://docs.yandex-team.ru/blackbox/concepts/intro-api
 */

#include <maps/libs/locale/include/locale.h>
#include <maps/wikimap/mapspro/libs/types/include/uid.h>
#include <maps/libs/common/include/retry.h>

#include <functional>
#include <optional>
#include <string>
#include <map>

namespace maps::wiki::blackbox {

using TvmTicketProvider = std::function<std::string()>;
using UID = types::TUid;


class Configuration
{
public:
    explicit Configuration(std::string host);

    const std::string& host() const;
    Configuration& setHost(std::string host);

private:
    std::string host_;
};


class UserInfo
{
public:
    UserInfo();

    const std::string& email() const;
    UserInfo& setEmail(std::string email);

    const std::optional<locale::Locale>& locale() const;
    UserInfo& setLocale(std::optional<locale::Locale> locale);

    const std::optional<std::string>& username() const;
    UserInfo& setUsername(std::optional<std::string> username);

    UID uid() const;
    UserInfo& setUid(UID uid);

private:
    std::string email_;
    std::optional<locale::Locale> locale_;
    std::optional<std::string> username_;
    UID uid_;
};


class IGateway
{
public:
    virtual bool isEmailValid(const std::string& email, UID uid) const = 0;
    virtual std::optional<UserInfo> defaultUserInfo(UID uid) const = 0;

    virtual ~IGateway() = default;
};

class Gateway: public IGateway
{
public:
    Gateway(
        Configuration configuration,
        const maps::common::RetryPolicy& retryPolicy,
        TvmTicketProvider tvmTicketProvider = TvmTicketProvider());

    /**
     * method=userinfo with emails=testone param
     * verifies email
     */
    bool isEmailValid(const std::string& email, UID uid) const override;

    /**
     * @brief method=userinfo with emails=getdefault
     * fetches user info with default email
     */
    std::optional<UserInfo> defaultUserInfo(UID uid) const override;

private:
    Configuration configuration_;
    maps::common::RetryPolicy retryPolicy_;
    TvmTicketProvider tvmTicketProvider_;
};

} // namespace maps::wiki::blackbox
