#include "calendar.h"

#include <maps/wikimap/mapspro/libs/masstransit/masstransit.h>

#include <sstream>

namespace maps::wiki::masstransit {

namespace {

const common::Date JANUARY_01 = common::Date(common::YEAR_ANY, common::Month(1), common::Day(1));
const common::Date DECEMBER_31 = common::Date(common::YEAR_ANY, common::Month(12), common::Day(31));

const std::string STR_ASTERISK = "*";

common::Schedule
readSchedule(const pqxx::row& tuple)
{
    auto startDate = getAttr<std::string>(tuple, attr::DATE_START, "");
    auto endDate = getAttr<std::string>(tuple, attr::DATE_END, "");
    auto weekdays = static_cast<common::WeekdayFlags>(getAttr<int>(
        tuple, attr::WEEK_DAY, static_cast<int>(common::WeekdayFlags::All)));

    auto startTime = getAttr<std::string>(tuple, attr::TIME_START, "");
    auto endTime = getAttr<std::string>(tuple, attr::TIME_END, "");
    auto frequency = getAttr<size_t>(tuple, attr::FREQ, 0) * common::SECONDS_PER_MINUTE;

    auto departureTime = getAttr<std::string>(tuple, attr::DEPARTURE_TIME, "");

    if (departureTime.empty()) {
        return common::Schedule(
            startDate, endDate, weekdays,
            startTime, endTime, frequency);
    } else {
        return common::Schedule(
            startDate, endDate, weekdays,
            {departureTime});
    }
}

std::string
datesToMtr(
    const common::Date& startDate,
    const common::Date& endDate)
{
    if (startDate == JANUARY_01 && endDate == DECEMBER_31) {
        return STR_ASTERISK;
    }

    std::ostringstream os;
    os << startDate;
    if (startDate != endDate) {
        os << "-" << endDate;
    }
    return os.str();
}

std::string
weekdaysToMtr(common::WeekdayFlags weekdays)
{
    if (weekdays == common::WeekdayFlags::All ||
        weekdays == common::WeekdayFlags::None)
    {
        return STR_ASTERISK;
    }

    std::string mtrWeekdays;
    for (int dayOfWeek = 1; dayOfWeek <= common::DAYS_PER_WEEK; ++dayOfWeek) {
        if ((weekdays & common::weekdayFromOrdinal(dayOfWeek)) !=
            common::WeekdayFlags::None)
        {
            if (!mtrWeekdays.empty()) {
                mtrWeekdays += ",";
            }
            mtrWeekdays += std::to_string(dayOfWeek);
        }
    }
    return mtrWeekdays;
}

} // namespace

YmapsdfCalendarHandler::YmapsdfCalendarHandler(Masstransit& masstransit)
    : YmapsdfObjectHandler(masstransit, masstransit.calendars)
{ }

std::vector<FtType>
YmapsdfCalendarHandler::ftTypes() const
{
    return {FtType::TransportFreqDt};
}

StringVector
YmapsdfCalendarHandler::attrs() const
{
    return {
        attr::DATE_START,
        attr::DATE_END,
        attr::WEEK_DAY,
        attr::TIME_START,
        attr::TIME_END,
        attr::FREQ,
        attr::DEPARTURE_TIME,
        attr::ALTERNATIVE
    };
}

StringVector
YmapsdfCalendarHandler::rolesToMasters() const
{
    return {role::APPLIED_TO};
}

std::string
YmapsdfCalendarHandler::extraSelectColumns() const
{
    return
        ", ("
            "SELECT ft_type_id "
            "FROM ft_ft, ft AS master_ft "
            "WHERE ft_ft.master_ft_id = master_ft.ft_id "
            "AND ft_ft.slave_ft_id = ft.ft_id"
        ") AS p_ft_type_id";
}

void
YmapsdfCalendarHandler::addObject(const pqxx::row& tuple)
{
    if (getAttr<FtType>(tuple, ymapsdf::P_FT_TYPE_ID) == FtType::TransportSystem) {
        return;
    }

    const auto threadId = getAttr<DBID>(tuple, role::APPLIED_TO);

    if (!masstransit_.calendars.count(threadId)) {
        auto calendar = std::make_shared<Calendar>();
        calendar->threadId = threadId;
        insert(threadId, calendar);
    }
    auto& calendar = masstransit_.calendars[threadId];

    auto schedule = readSchedule(tuple);
    auto isAlternative = getAttr<bool>(tuple, attr::ALTERNATIVE, false);

    if (isAlternative) {
        calendar.altMtrSchedules = common::unionSchedules(
            calendar.altMtrSchedules,
            {schedule});
    } else {
        calendar.mtrSchedules = common::unionSchedules(
            calendar.mtrSchedules,
            {schedule});
    }
}

void
YmapsdfCalendarHandler::update()
{
    for (auto& object : masstransit_.calendars()) {
        auto& calendar = dynamic_cast<Calendar&>(*object);

        calendar.mtrSchedules = common::replaceSchedules(
            calendar.mtrSchedules,
            calendar.altMtrSchedules);
        calendar.altMtrSchedules.clear();

        const auto& thread = masstransit_.threads[calendar.threadId];
        const auto& route = masstransit_.routes[thread.routeId];
        const auto& transportOperator = masstransit_.operators[route.operatorId];

        for (auto& schedule : calendar.mtrSchedules) {
            if (!schedule.departureTimes().empty()) {
                continue;
            }
            if (!schedule.startTime() || !schedule.endTime()) {
                schedule.setTimeInterval(
                    *transportOperator.schedule.startTime(),
                    *transportOperator.schedule.endTime());
            }
            if (schedule.frequency() == 0) {
                schedule.setFrequency(transportOperator.schedule.frequency());
            }
        }
    }

    for (auto& object : masstransit_.travelTimes()) {
        auto& travelTime = dynamic_cast<TravelTime&>(*object);

        size_t lastSeqId = 0;
        TravelTimeDatum* lastTime = nullptr;

        for (auto& travelTimeDatum : travelTime.data) {
            if (travelTimeDatum.seqId > lastSeqId) {
                lastSeqId = travelTimeDatum.seqId;
                lastTime = &travelTimeDatum;
            }
        }

        REQUIRE(lastTime,
            "Invalid travel times in thread_id = " << travelTime.threadId);
        lastTime->departureTimeSec = 0;
    }
}

MtrCalendarHandler::MtrCalendarHandler(Masstransit& masstransit)
    : MtrObjectHandler(masstransit, masstransit.calendars)
{ }

void
MtrCalendarHandler::writeObject(StreamMap& ostreams, const Object& object)
{
    const auto& calendar = dynamic_cast<const Calendar&>(object);
    const auto threadId = calendar.threadId;
    if (!masstransit_.threads.count(threadId)) {
        return;
    }

    auto& ostreamCalendar = ostreams[filename::CALENDAR];
    auto& ostreamFrequency = ostreams[filename::FREQUENCY];
    auto& ostreamTimetable = ostreams[filename::TIMETABLE];

    for (const auto& schedule : calendar.mtrSchedules) {
        for (const auto& dayRange : schedule.dayRanges()) {
            const auto scheduleId = writeCalendar(ostreamCalendar, threadId,
                dayRange, schedule.weekdays());
            if (schedule.departureTimes().empty()) {
                writeFrequency(ostreamFrequency, threadId, scheduleId,
                    *schedule.startTime(), *schedule.endTime(), schedule.frequency());
            } else {
                for (const auto& depTime : schedule.departureTimes()) {
                    writeTimetable(ostreamTimetable, threadId, scheduleId, depTime);
                }
            }
        }
    }
}

DBID
MtrCalendarHandler::writeCalendar(
    std::ofstream& ostream,
    DBID threadId,
    const common::IntRange& dayRange,
    const common::WeekdayFlags weekdays)
{
    const auto startDate = common::Date(dayRange.first);
    const auto endDate = common::Date(dayRange.second);

    MtrCalendarData calendarData{threadId, startDate, endDate, weekdays};
    auto iter = usedCalendars_.find(calendarData);
    if (iter != usedCalendars_.end()) {
        return iter->second;
    }
    iter = usedCalendars_.insert(std::make_pair(calendarData, idMap().newDBID())).first;

    const auto threadMtrId = idMap().getMtrId(threadId);
    const auto scheduleId = iter->second;
    ostream
        << threadMtrId << FIELD_SEPARATOR
        << scheduleId << FIELD_SEPARATOR
        << STR_ASTERISK /*dd*/ << FIELD_SEPARATOR
        << datesToMtr(startDate, endDate) << FIELD_SEPARATOR
        << STR_ASTERISK /*ddmmyyyy*/ << FIELD_SEPARATOR
        << weekdaysToMtr(weekdays) << std::endl;

    return scheduleId;
}

void
MtrCalendarHandler::writeFrequency(
    std::ofstream& ostream,
    DBID threadId,
    DBID scheduleId,
    const common::Time& startTime,
    const common::Time& endTime,
    size_t frequency)
{
    ostream
        << scheduleId << FIELD_SEPARATOR
        << getTravelTimeMtrId(threadId) << FIELD_SEPARATOR
        << startTime << FIELD_SEPARATOR
        << (startTime <= endTime ? endTime : common::Time(endTime.hour() + 24, endTime.minute())) << FIELD_SEPARATOR
        << frequency << FIELD_SEPARATOR
        << "0" /*exact_times*/ << std::endl;
}

void
MtrCalendarHandler::writeTimetable(
    std::ofstream& ostream,
    DBID threadId,
    DBID scheduleId,
    const common::Time& departureTime)
{
    ostream
        << scheduleId << FIELD_SEPARATOR
        << getTravelTimeMtrId(threadId) << FIELD_SEPARATOR
        << departureTime << std::endl;
}

} // namespace maps::wiki::masstransit
